@extends('layouts.admin')

@section('title', 'Daftar Penawaran Produk')

@section('content')

    <!-- Breadcrumb -->
    @php
        $breadcrumbs = [
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Penawaran', 'url' => route('admin.product-inquiries.index')],
        ];
    @endphp
    <x-breadcrumb :items="$breadcrumbs" />

    <!-- Notifikasi Modern -->
    <x-admin-notification />

    @if (session('error'))
        <div x-data="{ show: true }" x-show="show" x-transition:enter="transition transform ease-out duration-500"
            x-transition:enter-start="-translate-y-10 opacity-0" x-transition:enter-end="translate-y-0 opacity-100"
            x-transition:leave="transition transform ease-in duration-500"
            x-transition:leave-start="translate-y-0 opacity-100" x-transition:leave-end="-translate-y-10 opacity-0"
            x-init="setTimeout(() => show = false, 4000)"
            class="fixed top-6 left-1/2 transform -translate-x-1/2 z-50 w-[90%] sm:w-[500px] bg-gradient-to-r from-red-500 to-pink-600 text-white shadow-2xl rounded-xl px-6 py-4 flex items-center justify-between space-x-3">
            <div class="flex items-center space-x-3">
                <div class="flex items-center justify-center w-8 h-8 bg-white/20 rounded-full">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5 text-white" viewBox="0 0 20 20"
                        fill="currentColor">
                        <path fill-rule="evenodd"
                            d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z"
                            clip-rule="evenodd" />
                    </svg>
                </div>
                <div class="flex-1 font-semibold text-sm sm:text-base">
                    {{ session('error') }}
                </div>
            </div>
            <button @click="show = false" class="hover:bg-white/20 rounded-full p-1 transition duration-200">
                <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5" fill="none" viewBox="0 0 24 24"
                    stroke="currentColor" stroke-width="2">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12" />
                </svg>
            </button>
        </div>
    @endif

    <!-- Header -->
    <div class="mb-6">
        <h1 class="text-2xl font-bold text-gray-800 dark:text-white mb-4">Daftar Penawaran</h1>

        <!-- Filter Section -->
        <div class="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-4 mb-4">
            <form method="GET" action="{{ route('admin.product-inquiries.index') }}" id="filterForm" class="space-y-4">
                <!-- Search Bar -->
                <div class="relative flex-1">
                    <input type="text" name="search" id="searchInput" value="{{ request('search') }}"
                        placeholder="Cari nama, email, produk..." class="form-input-search" />
                    <i
                        class="fas fa-search absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 dark:text-gray-500 pointer-events-none"></i>
                    @if (!empty(request('search')))
                        <button type="button" onclick="clearFilter('search')"
                            class="absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 dark:text-gray-500 hover:text-gray-600 dark:hover:text-gray-300">
                            <i class="fas fa-times"></i>
                        </button>
                    @endif
                </div>

                <!-- Filter Row -->
                <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
                    <!-- Status Filter -->
                    <div class="relative">
                        <select name="status" id="statusFilter" onchange="submitFilter()" class="form-select">
                            <option value="">Semua Status</option>
                            <option value="unread" {{ request('status') == 'unread' ? 'selected' : '' }}>Belum Dibaca
                            </option>
                            <option value="read" {{ request('status') == 'read' ? 'selected' : '' }}>Sudah Dibaca</option>
                        </select>
                        <i
                            class="fas fa-chevron-down absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 dark:text-gray-500 pointer-events-none text-xs"></i>
                    </div>

                    <!-- Tahap Filter -->
                    <div class="relative">
                        <select name="state" id="stateFilter" onchange="submitFilter()" class="form-select">
                            <option value="">Semua Tahap</option>
                            @php
                                $stateLabels = [
                                    'pending' => 'Menunggu',
                                    'supervisor_review' => 'Review Supervisor',
                                    'teknisi_review' => 'Review Teknisi',
                                    'sent_to_admin' => 'Dikirim ke Admin',
                                    'approved' => 'Disetujui',
                                    'customer_accepted' => 'Diterima Customer',
                                    'customer_counter' => 'Tawar Menawar',
                                    'converted_to_order' => 'Jadi Pesanan',
                                    'rejected' => 'Ditolak',
                                ];
                            @endphp
                            @foreach ($stateLabels as $key => $label)
                                <option value="{{ $key }}" {{ request('state') === $key ? 'selected' : '' }}>
                                    {{ $label }}</option>
                            @endforeach
                        </select>
                        <i
                            class="fas fa-chevron-down absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 dark:text-gray-500 pointer-events-none text-xs"></i>
                    </div>

                    <!-- Bulk Actions -->
                    <div class="relative">
                        <select id="bulkAction"
                            class="appearance-none w-full pl-10 pr-8 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors duration-200 text-sm">
                            <option value="">Bulk Actions</option>
                            <option value="mark-read">Tandai Dibaca</option>
                            <option value="mark-unread">Tandai Belum Dibaca</option>
                            <option value="delete">Hapus</option>
                        </select>
                        <i
                            class="fas fa-list absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 dark:text-gray-500 pointer-events-none"></i>
                        <i
                            class="fas fa-chevron-down absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 dark:text-gray-500 pointer-events-none text-xs"></i>
                    </div>

                    <!-- Action Buttons -->
                    <div class="flex items-center gap-3 w-full sm:w-auto">
                        <button type="button" onclick="executeBulkAction()" id="bulkActionBtn" disabled
                            class="action-btn-primary disabled:opacity-50 disabled:cursor-not-allowed">
                            <i class="fas fa-check mr-2"></i>Execute
                        </button>
                        <a href="{{ route('admin.product-inquiries.index') }}" class="action-btn-primary">
                            <i class="fas fa-refresh mr-2"></i>Refresh
                        </a>
                    </div>
                </div>

                <!-- Clear Filters Button -->
                @if (request('search') || request('status') || request('state'))
                    <div class="flex justify-end">
                        <button type="button" onclick="clearAllFilters()"
                            class="inline-flex items-center px-4 py-2 text-sm font-medium rounded-lg border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors duration-200">
                            <i class="fas fa-times mr-2"></i>
                            Reset Filter
                        </button>
                    </div>
                @endif
            </form>
        </div>
    </div>

    <!-- Active Filters Info -->
    @if (request('search') || request('status') || request('state'))
        <div class="mb-4 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-700 rounded-lg p-4">
            <div class="flex flex-wrap items-center gap-2">
                <span class="text-sm font-medium text-blue-800 dark:text-blue-200">Filter Aktif:</span>

                @if (request('search'))
                    <span
                        class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800 dark:bg-blue-800 dark:text-blue-200">
                        <i class="fas fa-search mr-1"></i>
                        Pencarian: "{{ request('search') }}"
                        <button onclick="clearFilter('search')" class="ml-2 hover:text-blue-600">
                            <i class="fas fa-times"></i>
                        </button>
                    </span>
                @endif

                @if (request('status'))
                    <span
                        class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800 dark:bg-green-800 dark:text-green-200">
                        <i class="fas fa-filter mr-1"></i>
                        Status: {{ request('status') == 'unread' ? 'Belum Dibaca' : 'Sudah Dibaca' }}
                        <button onclick="clearFilter('status')" class="ml-2 hover:text-green-600">
                            <i class="fas fa-times"></i>
                        </button>
                    </span>
                @endif

                @if (request('state'))
                    @php
                        $stateLabels = [
                            'pending' => 'Menunggu',
                            'supervisor_review' => 'Review Supervisor',
                            'teknisi_review' => 'Review Teknisi',
                            'sent_to_admin' => 'Dikirim ke Admin',
                            'approved' => 'Disetujui',
                            'customer_accepted' => 'Diterima Customer',
                            'customer_counter' => 'Tawar Menawar',
                            'converted_to_order' => 'Jadi Pesanan',
                            'rejected' => 'Ditolak',
                        ];
                        $stateLabel =
                            $stateLabels[request('state')] ?? ucwords(str_replace('_', ' ', request('state')));
                    @endphp
                    <span
                        class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-purple-100 text-purple-800 dark:bg-purple-800 dark:text-purple-200">
                        <i class="fas fa-tasks mr-1"></i>
                        Tahap: {{ $stateLabel }}
                        <button onclick="clearFilter('state')" class="ml-2 hover:text-purple-600">
                            <i class="fas fa-times"></i>
                        </button>
                    </span>
                @endif

                <button onclick="clearAllFilters()"
                    class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-red-100 text-red-800 dark:bg-red-800 dark:text-red-200 hover:bg-red-200 dark:hover:bg-red-700 transition-colors">
                    <i class="fas fa-times-circle mr-1"></i>
                    Hapus Semua Filter
                </button>
            </div>
        </div>
    @endif

    <!-- Table Container -->
    <div
        class="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden transition-colors duration-300">
        <div class="overflow-x-auto">
            <table class="w-full">
                <thead class="bg-gray-50 dark:bg-gray-700/50 border-b border-gray-200 dark:border-gray-600">
                    <tr>
                        <th class="w-12 px-4 py-3.5 text-center">
                            <input type="checkbox" id="selectAll" onchange="toggleSelectAll()"
                                class="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600">
                        </th>
                        <th
                            class="w-16 px-4 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                            No
                        </th>
                        <th
                            class="px-6 py-3.5 text-left text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                            Nama</th>
                        <th
                            class="px-6 py-3.5 text-left text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                            Email</th>
                        <th
                            class="px-6 py-3.5 text-left text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                            Produk</th>
                        <th
                            class="px-6 py-3.5 text-left text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                            Pesan</th>
                        <th
                            class="px-6 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                            Status</th>
                        <th
                            class="px-6 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                            Tahap</th>
                        <th
                            class="px-6 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                            Tanggal</th>
                        <th
                            class="px-6 py-3.5 text-center text-xs font-semibold uppercase tracking-wider text-gray-600 dark:text-gray-300">
                            Aksi</th>
                    </tr>
                </thead>

                <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                    @forelse ($inquiries as $inquiry)
                        <tr data-inquiry-id="{{ $inquiry->id }}" data-inquiry-name="{{ $inquiry->name }}"
                            data-inquiry-email="{{ $inquiry->email }}" data-inquiry-phone="{{ $inquiry->phone }}"
                            data-inquiry-message="{{ $inquiry->message }}"
                            data-inquiry-product="{{ $inquiry->product->product_title ?? 'Produk tidak ditemukan' }}"
                            data-inquiry-tracking="{{ $inquiry->tracking_code }}"
                            data-inquiry-date="{{ $inquiry->created_at->format('d M Y H:i') }}"
                            data-inquiry-type="{{ $inquiry->inquiry_type ?? 'catalog' }}"
                            data-is-read="{{ $inquiry->is_read ? 1 : 0 }}"
                            data-has-reply="{{ $inquiry->admin_reply ? 1 : 0 }}"
                            data-inquiry-status="{{ $inquiry->status }}"
                            data-materials-breakdown="{{ $inquiry->materials_breakdown ? json_encode($inquiry->materials_breakdown) : '[]' }}"
                            data-spareparts-breakdown="{{ $inquiry->spareparts_breakdown ? json_encode($inquiry->spareparts_breakdown) : '[]' }}"
                            data-labor-cost="{{ $inquiry->labor_cost ?? 0 }}"
                            data-labor-cost-total="{{ $inquiry->labor_cost_total ?? $inquiry->labor_cost ?? 0 }}"
                            data-materials-total="{{ $inquiry->materials_total_cost ?? 0 }}"
                            data-spareparts-total="{{ $inquiry->spareparts_total_cost ?? 0 }}"
                            data-total-cost="{{ $inquiry->total_cost ?? 0 }}"
                            data-engineer-suggested="{{ $inquiry->engineer_suggested_price ?? 0 }}"
                            data-estimated-days="{{ $inquiry->estimated_days ?? 0 }}"
                            data-teknisi-note="{{ htmlspecialchars($inquiry->teknisi_note ?? '', ENT_QUOTES, 'UTF-8') }}"
                            data-teknisi-name="{{ $inquiry->teknisi->name ?? '' }}"
                            data-teknisi-estimated-at="{{ $inquiry->teknisi_estimated_at ? $inquiry->teknisi_estimated_at->format('d M Y H:i') : '' }}"
                            data-admin-revision-note="{{ htmlspecialchars($inquiry->admin_revision_note ?? '', ENT_QUOTES, 'UTF-8') }}"
                            data-admin-revision-at="{{ $inquiry->admin_revision_at ? $inquiry->admin_revision_at->format('d M Y H:i') : '' }}"
                            class="hover:bg-blue-50/50 dark:hover:bg-gray-700/80 transition-colors duration-200 cursor-pointer">
                            <td class="px-4 py-4 text-center">
                                <input type="checkbox"
                                    class="inquiry-checkbox w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                    value="{{ $inquiry->id }}" onchange="updateBulkActionButton()">
                            </td>
                            <td class="px-4 py-4 text-sm text-gray-900 dark:text-gray-100 text-center align-middle">
                                {{ $loop->iteration + $inquiries->firstItem() - 1 }}
                            </td>

                            <td class="px-6 py-4 text-sm text-gray-900 dark:text-gray-100 font-medium align-middle">
                                <div class="flex items-center space-x-2">
                                    {{ $inquiry->name }}
                                    @if (!$inquiry->is_read)
                                        <span
                                            class="inline-flex items-center justify-center w-2 h-2 bg-red-500 rounded-full animate-pulse"
                                            title="Pesanan Baru"></span>
                                    @endif
                                </div>
                            </td>

                            <td
                                class="px-6 py-4 text-sm text-gray-600 dark:text-gray-400 align-middle break-words max-w-xs">
                                {{ $inquiry->email }}
                            </td>

                            <td class="px-6 py-4 text-sm text-gray-900 dark:text-gray-100 align-middle">
                                <span class="truncate max-w-xs block"
                                    title="{{ $inquiry->product->product_title ?? 'Produk tidak ditemukan' }}">
                                    {{ Str::limit($inquiry->product->product_title ?? 'Produk tidak ditemukan', 30) }}
                                </span>
                            </td>

                            <td
                                class="px-6 py-4 text-sm text-gray-700 dark:text-gray-300 align-middle break-words max-w-sm">
                                @php
                                    $raw = $inquiry->message ?? '';
                                    $lines = preg_split("/(\r\n|\n|\r)/", $raw);
                                    $type = null;
                                    $addons = [];
                                    $hasAddonsLine = false;
                                    if (!empty($lines)) {
                                        $first = $lines[0] ?? '';
                                        $second = $lines[1] ?? '';
                                        if (substr($first, 0, 14) === '[REQUEST_TYPE]') {
                                            $type = trim(substr($first, 14));
                                        }
                                        if (substr($second, 0, 8) === '[ADDONS]') {
                                            $hasAddonsLine = true;
                                            $matches = [];
                                            if (preg_match('/^\[ADDONS\]\s*(\[[\s\S]*\])$/', $second, $matches)) {
                                                $decoded = json_decode($matches[1], true);
                                                if (is_array($decoded)) {
                                                    $addons = $decoded;
                                                }
                                            }
                                        }
                                    }
                                    $startIdx = 0;
                                    if ($type) {
                                        $startIdx++;
                                    }
                                    if ($hasAddonsLine) {
                                        $startIdx++;
                                    }
                                    if (!empty($lines) && isset($lines[$startIdx]) && trim($lines[$startIdx]) === '') {
                                        $startIdx++;
                                    }
                                    $plain = implode("\n", array_slice($lines, $startIdx));

                                    $badgeText =
                                        $type === 'custom_addon'
                                            ? 'Custom Add-on'
                                            : ($type === 'custom_new'
                                                ? 'Custom Baru'
                                                : ($type
                                                    ? 'Katalog'
                                                    : null));
                                    $badgeCls =
                                        $type === 'custom_addon'
                                            ? 'bg-yellow-100 text-yellow-800'
                                            : ($type === 'custom_new'
                                                ? 'bg-red-100 text-red-800'
                                                : 'bg-blue-100 text-blue-800');
                                @endphp
                                <div class="space-y-1">
                                    @if ($badgeText)
                                        <span
                                            class="inline-flex items-center px-2 py-0.5 text-xs font-semibold rounded-full {{ $badgeCls }}">{{ $badgeText }}</span>
                                    @endif
                                    @if (!empty($addons))
                                        @php
                                            $preview = collect($addons)
                                                ->take(2)
                                                ->map(function ($a) {
                                                    $name = is_array($a) ? $a['name'] ?? '' : '';
                                                    $value = is_array($a) ? $a['value'] ?? '' : '';
                                                    return trim(($name ?: '-') . ': ' . ($value ?: '-'));
                                                })
                                                ->implode('; ');
                                            $moreCount = max(count($addons) - 2, 0);
                                        @endphp
                                        <div class="text-xs text-gray-600 dark:text-gray-400 truncate max-w-sm"
                                            title="{{ $preview }}@if ($moreCount > 0) (+{{ $moreCount }} lagi) @endif">
                                            <i class="fas fa-tools mr-1"></i>Spesifikasi: {{ $preview }}@if ($moreCount > 0)
                                                (+{{ $moreCount }} lagi)
                                            @endif
                                        </div>
                                    @endif
                                    <p class="text-gray-700 dark:text-gray-300 text-sm"
                                        style="display:-webkit-box;-webkit-line-clamp:2;-webkit-box-orient:vertical;overflow:hidden"
                                        title="{{ $plain }}">{{ $plain }}</p>
                                </div>
                            </td>

                            <td class="px-6 py-4 text-center align-middle">
                                @if ($inquiry->is_read)
                                    <span
                                        class="inline-flex items-center gap-1.5 px-2.5 py-1 text-xs font-medium rounded-md bg-green-100 dark:bg-green-900/40 text-green-700 dark:text-green-300">
                                        <i class="fas fa-check-circle text-xs"></i>
                                        <span>Dibaca</span>
                                    </span>
                                @else
                                    <span
                                        class="inline-flex items-center gap-1.5 px-2.5 py-1 text-xs font-medium rounded-md bg-red-100 dark:bg-red-900/40 text-red-700 dark:text-red-300">
                                        <i class="fas fa-eye-slash text-xs"></i>
                                        <span>Belum Dibaca</span>
                                    </span>
                                @endif
                            </td>

                            <td class="px-6 py-4 text-center align-middle">
                                @php
                                    $state = $inquiry->status ?? 'pending';
                                    $stateLabels = [
                                        'pending' => 'Menunggu',
                                        'supervisor_review' => 'Review Supervisor',
                                        'teknisi_review' => 'Review Teknisi',
                                        'admin_review' => 'Review Admin',
                                        'sent_to_admin' => 'Dikirim ke Admin',
                                        'sent_to_customer' => 'Dikirim ke Customer',
                                        'approved' => 'Disetujui',
                                        'customer_accepted' => 'Diterima Customer',
                                        'customer_counter' => 'Tawar Menawar',
                                        'converted_to_order' => 'Jadi Pesanan',
                                        'rejected' => 'Ditolak',
                                    ];
                                    $map = [
                                        'pending' => 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300',
                                        'supervisor_review' =>
                                            'bg-purple-100 dark:bg-purple-900/40 text-purple-700 dark:text-purple-300',
                                        'teknisi_review' =>
                                            'bg-indigo-100 dark:bg-indigo-900/40 text-indigo-700 dark:text-indigo-300',
                                        'admin_review' =>
                                            'bg-cyan-100 dark:bg-cyan-900/40 text-cyan-700 dark:text-cyan-300',
                                        'sent_to_admin' =>
                                            'bg-blue-100 dark:bg-blue-900/40 text-blue-700 dark:text-blue-300',
                                        'sent_to_customer' =>
                                            'bg-teal-100 dark:bg-teal-900/40 text-teal-700 dark:text-teal-300',
                                        'approved' =>
                                            'bg-green-100 dark:bg-green-900/40 text-green-700 dark:text-green-300',
                                        'customer_accepted' =>
                                            'bg-green-100 dark:bg-green-900/40 text-green-700 dark:text-green-300',
                                        'customer_counter' =>
                                            'bg-yellow-100 dark:bg-yellow-900/40 text-yellow-700 dark:text-yellow-300',
                                        'converted_to_order' =>
                                            'bg-purple-100 dark:bg-purple-900/40 text-purple-700 dark:text-purple-300',
                                        'rejected' => 'bg-red-100 dark:bg-red-900/40 text-red-700 dark:text-red-300',
                                    ];
                                    $cls =
                                        $map[$state] ?? 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300';
                                    $label = $stateLabels[$state] ?? ucwords(str_replace('_', ' ', $state));
                                @endphp
                                <span
                                    class="inline-flex items-center gap-1.5 px-2.5 py-1 text-xs font-medium rounded-md {{ $cls }}">
                                    <i class="fas fa-circle text-[6px]"></i>
                                    <span>{{ $label }}</span>
                                </span>
                            </td>
                            <td class="px-6 py-4 text-sm text-gray-500 dark:text-gray-400 text-center align-middle">
                                {{ $inquiry->created_at->format('d M Y H:i') }}
                            </td>

                            <td class="px-6 py-4 text-center align-middle">
                                <div class="flex items-center justify-center space-x-2">
                                    <!-- Tombol Lihat Detail -->
                                    <a href="#" type="button" data-id="{{ $inquiry->id }}"
                                        onclick="openInquiryModalFromBtn(this); return false;"
                                        class="inline-flex items-center justify-center w-9 h-9 text-indigo-600 dark:text-indigo-400 hover:text-indigo-700 dark:hover:text-indigo-300 hover:bg-indigo-50 dark:hover:bg-indigo-900/20 rounded transition-colors duration-200"
                                        title="Lihat Detail">
                                        <i class="fas fa-eye text-base"></i>
                                    </a>

                                    @php $state = $inquiry->status ?? 'pending'; @endphp
                                    @if (in_array($state, ['pending', 'customer_counter']))
                                        <!-- Teruskan ke Teknisi (untuk produk custom) -->
                                        <button type="button" data-id="{{ $inquiry->id }}"
                                            onclick="forwardToTeknisiFromBtn(this)"
                                            class="inline-flex items-center justify-center w-9 h-9 text-orange-600 dark:text-orange-400 hover:text-orange-700 dark:hover:text-orange-300 hover:bg-orange-50 dark:hover:bg-orange-900/20 rounded transition-colors duration-200"
                                            title="Teruskan ke Teknisi (untuk estimasi produk custom)">
                                            <i class="fas fa-tools text-base"></i>
                                        </button>
                                    @endif

                                    @if (in_array($state, ['pending', 'admin_review', 'customer_counter']))
                                        <!-- Teruskan ke Supervisor -->
                                        <button type="button" data-id="{{ $inquiry->id }}"
                                            onclick="forwardToSupervisorFromBtn(this)"
                                            class="inline-flex items-center justify-center w-9 h-9 text-purple-600 dark:text-purple-400 hover:text-purple-700 dark:hover:text-purple-300 hover:bg-purple-50 dark:hover:bg-purple-900/20 rounded transition-colors duration-200"
                                            title="Teruskan ke Supervisor untuk review RAB & set koridor harga">
                                            <i class="fas fa-user-tie text-base"></i>
                                        </button>
                                    @endif

                                    <!-- Tombol WhatsApp (Chat Langsung) -->
                                    <a href="https://wa.me/{{ $inquiry->phone }}" target="_blank"
                                        class="inline-flex items-center justify-center w-9 h-9 text-green-600 dark:text-green-400 hover:text-green-700 dark:hover:text-green-300 hover:bg-green-50 dark:hover:bg-green-900/20 rounded transition-colors duration-200"
                                        title="Chat WhatsApp">
                                        <i class="fab fa-whatsapp text-base"></i>
                                    </a>

                                    <!-- Tombol Email -->
                                    <button type="button" data-id="{{ $inquiry->id }}"
                                        onclick="openReplyModal({{ $inquiry->id }})"
                                        class="inline-flex items-center justify-center w-9 h-9 text-blue-600 dark:text-blue-400 hover:text-blue-700 dark:hover:text-blue-300 hover:bg-blue-50 dark:hover:bg-blue-900/20 rounded transition-colors duration-200"
                                        title="Kirim Email">
                                        <i class="fas fa-envelope text-base"></i>
                                    </button>

                                    @if (!is_null($inquiry->final_offer_price))
                                        <!-- Tombol Copy Link Dokumen Penawaran -->
                                        <button type="button"
                                            data-preview-url="{{ route('admin.print.quotation.offer.preview', $inquiry) }}"
                                            onclick="copyPreviewLinkFromBtn(this)"
                                            class="inline-flex items-center justify-center w-9 h-9 text-gray-600 dark:text-gray-300 hover:text-gray-800 dark:hover:text-gray-100 hover:bg-gray-50 dark:hover:bg-gray-900/20 rounded transition-colors duration-200"
                                            title="Copy Link Dokumen">
                                            <i class="fas fa-link text-base"></i>
                                        </button>
                                    @endif

                                    <!-- Tombol Mark as Read (hanya jika belum dibaca) -->
                                    @if (!$inquiry->is_read)
                                        <button type="button" data-id="{{ $inquiry->id }}"
                                            onclick="markAsReadFromBtn(this)"
                                            class="inline-flex items-center justify-center w-9 h-9 text-orange-600 dark:text-orange-400 hover:text-orange-700 dark:hover:text-orange-300 hover:bg-orange-50 dark:hover:bg-orange-900/20 rounded transition-colors duration-200"
                                            title="Tandai Dibaca">
                                            <i class="fas fa-check text-base"></i>
                                        </button>
                                    @endif

                                    @if (($inquiry->status ?? '') === 'sent_to_admin')
                                        @php
                                            $finalPayload = [
                                                'id' => $inquiry->id,
                                                'productTitle' => $inquiry->product->product_title ?? '',
                                                'floor' => (float) ($inquiry->supervisor_floor_price ?? 0),
                                                'target' => (float) ($inquiry->supervisor_target_price ?? 0),
                                                'ceiling' => (float) ($inquiry->supervisor_ceiling_price ?? 0),
                                                'validity' => $inquiry->validity_until
                                                    ? \Carbon\Carbon::parse($inquiry->validity_until)->format('Y-m-d')
                                                    : '',
                                                'internalCost' => (float) ($inquiry->internal_cost ?? 0),
                                                'marginPercent' => $inquiry->margin_percent,
                                            ];
                                        @endphp
                                        <button type="button"
                                            class="inline-flex items-center justify-center w-9 h-9 text-green-600 dark:text-green-400 hover:text-green-700 dark:hover:text-green-300 hover:bg-green-50 dark:hover:bg-green-900/20 rounded transition-colors duration-200"
                                            data-id="{{ $finalPayload['id'] }}"
                                            data-title="{{ $finalPayload['productTitle'] }}"
                                            data-floor="{{ $finalPayload['floor'] }}"
                                            data-target="{{ $finalPayload['target'] }}"
                                            data-ceiling="{{ $finalPayload['ceiling'] }}"
                                            data-validity="{{ $finalPayload['validity'] }}"
                                            data-internalcost="{{ $finalPayload['internalCost'] }}"
                                            data-margin="{{ $finalPayload['marginPercent'] ?? '' }}"
                                            onclick="openFinalOfferModalFromData(this)" title="Kirim Final Offer">
                                            <i class="fas fa-paper-plane text-base"></i>
                                        </button>
                                    @endif

                                    @if (!empty($inquiry->final_offer_price) && !empty($inquiry->phone))
                                        @php
                                            $waNumber = preg_replace('/\D/', '', $inquiry->phone);
                                            $waText =
                                                'Halo ' .
                                                $inquiry->name .
                                                ",\n\n" .
                                                'Penawaran harga final untuk ' .
                                                ($inquiry->product->product_title ?? 'produk') .
                                                ' adalah Rp ' .
                                                number_format($inquiry->final_offer_price, 0, ',', '.') .
                                                ".\nLihat dokumen: " .
                                                route('admin.print.quotation.offer.preview', $inquiry);
                                            $waLink = 'https://wa.me/' . $waNumber . '?text=' . urlencode($waText);
                                        @endphp
                                        <a href="{{ $waLink }}" target="_blank"
                                            class="inline-flex items-center justify-center w-9 h-9 text-green-600 dark:text-green-400 hover:text-green-700 dark:hover:text-green-300 hover:bg-green-50 dark:hover:bg-green-900/20 rounded transition-colors duration-200 relative"
                                            title="Kirim Penawaran Final via WhatsApp">
                                            <i class="fab fa-whatsapp text-base"></i>
                                        </a>
                                    @endif

                                    @if (
                                        ($inquiry->status ?? '') === 'customer_accepted' &&
                                            !empty($inquiry->final_offer_price) &&
                                            is_null($inquiry->order_id))
                                        <button type="button" data-id="{{ $inquiry->id }}"
                                            data-name="{{ $inquiry->name }}" data-email="{{ $inquiry->email }}"
                                            data-product="{{ $inquiry->product->product_title ?? '' }}"
                                            data-price="{{ (float) ($inquiry->final_offer_price ?? 0) }}"
                                            onclick="openConvertOrderModalFromBtn(this)"
                                            class="inline-flex items-center justify-center w-9 h-9 text-blue-600 dark:text-blue-400 hover:text-blue-700 dark:hover:text-blue-300 hover:bg-blue-50 dark:hover:bg-blue-900/20 rounded transition-colors duration-200"
                                            title="Buat Order">
                                            <i class="fas fa-shopping-cart text-base"></i>
                                        </button>
                                    @endif

                                    @if (($inquiry->status ?? '') === 'converted_to_order' && $inquiry->order_id)
                                        <a href="{{ route('admin.master.orders.show', $inquiry->order_id) }}"
                                            target="_blank"
                                            class="inline-flex items-center justify-center w-9 h-9 text-purple-600 dark:text-purple-400 hover:text-purple-700 dark:hover:text-purple-300 hover:bg-purple-50 dark:hover:bg-purple-900/20 rounded transition-colors duration-200"
                                            title="Lihat Order #{{ $inquiry->order_id }}">
                                            <i class="fas fa-external-link-alt text-base"></i>
                                        </a>
                                    @endif

                                    <!-- Tombol Hapus -->
                                    <button type="button" data-id="{{ $inquiry->id }}"
                                        onclick="confirmDeleteInquiry({{ $inquiry->id }})"
                                        class="inline-flex items-center justify-center w-9 h-9 text-red-600 dark:text-red-400 hover:text-red-700 dark:hover:text-red-300 hover:bg-red-50 dark:hover:bg-red-900/20 rounded transition-colors duration-200"
                                        title="Hapus">
                                        <i class="fas fa-trash text-base"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                        <!-- Form Hapus Tersembunyi -->
                        <form id="delete-form-{{ $inquiry->id }}"
                            action="{{ route('admin.product-inquiries.destroy', $inquiry->id) }}" method="POST"
                            class="hidden">
                            @csrf
                            @method('DELETE')
                        </form>
                        @empty
                            <tr>
                                <td colspan="10" class="px-6 py-16 text-center">
                                    <div class="flex flex-col items-center">
                                        <div
                                            class="w-16 h-16 bg-gray-100 dark:bg-gray-700 rounded-full flex items-center justify-center mb-4">
                                            <i class="fas fa-envelope-open-text text-2xl text-gray-400 dark:text-gray-500"></i>
                                        </div>
                                        <p class="text-gray-500 dark:text-gray-400 text-lg font-medium">Belum ada penawaran
                                            produk</p>
                                        <p class="text-gray-400 dark:text-gray-500 text-sm mt-1">Penawaran produk akan muncul
                                            di sini</p>
                                    </div>
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
            {{-- Pagination (Inside Container, Tipis) --}}
            @if ($inquiries->hasPages())
                <div
                    class="flex items-center justify-between px-4 py-3 border-t border-gray-200 dark:border-gray-700 text-sm">
            
                    {{-- Info --}}
                    <div class="text-gray-600 dark:text-gray-400">
                        Menampilkan
                        <span class="font-medium">{{ $inquiries->firstItem() }}</span>
                        -
                        <span class="font-medium">{{ $inquiries->lastItem() }}</span>
                        dari
                        <span class="font-medium">{{ $inquiries->total() }}</span>
                        data
                    </div>
            
                    {{-- Pagination --}}
                    <div>
                        {{ $inquiries->onEachSide(1)->links() }}
                    </div>
                </div>
            @endif

        </div>

        <!-- Final Offer Modal -->
        <div id="finalOfferModal"
            class="fixed inset-0 bg-black/60 backdrop-blur-sm hidden items-center justify-center z-[9999] transition-all duration-300 ease-in-out">
            <div id="finalOfferModalContent"
                class="bg-white dark:bg-gray-800 rounded-lg border border-gray-300 dark:border-gray-600 shadow-2xl max-w-xl w-full mx-4 p-0 overflow-hidden transform transition-all duration-300 ease-in-out max-h-[90vh] flex flex-col">
                <!-- Header -->
                <div class="bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700 px-6 py-4 flex-shrink-0">
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-3">
                            <div class="w-10 h-10 bg-gray-100 dark:bg-gray-700 rounded-lg flex items-center justify-center">
                                <i class="fas fa-paper-plane text-gray-600 dark:text-gray-300 text-lg"></i>
                            </div>
                            <h2 class="text-xl font-semibold text-gray-900 dark:text-white">Final Offer ke Customer</h2>
                        </div>
                        <button onclick="closeFinalOfferModal()"
                            class="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 transition-colors duration-200 p-1 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700">
                            <i class="fas fa-times text-lg"></i>
                        </button>
                    </div>
                </div>

                <!-- Content - Scrollable -->
                <div class="p-6 overflow-y-auto flex-1">
                    <form id="finalOfferForm">
                        @csrf
                        <input type="hidden" id="final-inquiry-id" />
                        <div class="space-y-4">
                            <!-- Product Info -->
                            <div
                                class="bg-gray-50 dark:bg-gray-700/30 rounded-lg border border-gray-200 dark:border-gray-600 p-4">
                                <label class="block text-xs font-semibold text-gray-700 dark:text-gray-300 mb-2">Produk</label>
                                <input type="text" id="final-product-title"
                                    class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-700 dark:text-white text-sm"
                                    disabled />
                            </div>

                            <!-- Price Range Info -->
                            <div
                                class="bg-gray-50 dark:bg-gray-700/30 rounded-lg border border-gray-200 dark:border-gray-600 p-4">
                                <label class="block text-xs font-semibold text-gray-700 dark:text-gray-300 mb-3">Koridor Harga
                                    Supervisor</label>
                                <div class="grid grid-cols-3 gap-3">
                                    <div
                                        class="text-center p-2 bg-white dark:bg-gray-800 rounded border border-gray-200 dark:border-gray-600">
                                        <div class="text-xs text-gray-500 dark:text-gray-400 mb-1">Floor</div>
                                        <div id="final-floor" class="text-sm font-semibold text-gray-900 dark:text-gray-100">-
                                        </div>
                                    </div>
                                    <div
                                        class="text-center p-2 bg-white dark:bg-gray-800 rounded border border-gray-200 dark:border-gray-600">
                                        <div class="text-xs text-gray-500 dark:text-gray-400 mb-1">Target</div>
                                        <div id="final-target" class="text-sm font-semibold text-gray-900 dark:text-gray-100">
                                            -</div>
                                    </div>
                                    <div
                                        class="text-center p-2 bg-white dark:bg-gray-800 rounded border border-gray-200 dark:border-gray-600">
                                        <div class="text-xs text-gray-500 dark:text-gray-400 mb-1">Ceiling</div>
                                        <div id="final-ceiling"
                                            class="text-sm font-semibold text-gray-900 dark:text-gray-100">-</div>
                                    </div>
                                </div>
                                <div class="grid grid-cols-2 gap-3 mt-3">
                                    <div
                                        class="p-2 bg-white dark:bg-gray-800 rounded border border-gray-200 dark:border-gray-600">
                                        <div class="text-xs text-gray-500 dark:text-gray-400 mb-1">Biaya Internal</div>
                                        <div id="final-internal"
                                            class="text-sm font-semibold text-gray-900 dark:text-gray-100">-</div>
                                    </div>
                                    <div
                                        class="p-2 bg-white dark:bg-gray-800 rounded border border-gray-200 dark:border-gray-600">
                                        <div class="text-xs text-gray-500 dark:text-gray-400 mb-1">Margin</div>
                                        <div id="final-margin" class="text-sm font-semibold text-gray-900 dark:text-gray-100">
                                            -</div>
                                    </div>
                                </div>
                            </div>

                            <!-- Final Price Input -->
                            <div>
                                <label class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                    <i class="fas fa-dollar-sign mr-2 text-gray-500"></i>Harga Final
                                </label>
                                <input type="number" step="0.01" id="final-price"
                                    class="w-full px-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white transition-colors duration-200"
                                    placeholder="Masukkan harga final" required />
                                <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                    <i class="fas fa-info-circle mr-1"></i>Harga harus berada dalam koridor supervisor (Floor -
                                    Ceiling)
                                </p>
                            </div>

                            <!-- Notes -->
                            <div>
                                <label class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                    <i class="fas fa-sticky-note mr-2 text-gray-500"></i>Catatan untuk Customer
                                </label>
                                <textarea id="final-notes" rows="4"
                                    class="w-full px-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 resize-none dark:bg-gray-700 dark:text-white transition-colors duration-200"
                                    placeholder="Syarat pembayaran, masa berlaku, lead time, dll"></textarea>
                            </div>

                            <!-- Validity -->
                            <div
                                class="bg-gray-50 dark:bg-gray-700/30 rounded-lg border border-gray-200 dark:border-gray-600 p-3">
                                <div class="flex items-center">
                                    <i class="fas fa-calendar-alt text-gray-600 dark:text-gray-400 mr-2"></i>
                                    <span class="text-xs text-gray-600 dark:text-gray-400">Berlaku hingga:</span>
                                    <span id="final-validity"
                                        class="text-xs font-semibold text-gray-900 dark:text-gray-100 ml-2">-</span>
                                </div>
                            </div>

                        </div>
                    </form>
                </div>

                <!-- Footer - Fixed -->
                <div class="bg-white dark:bg-gray-800 border-t border-gray-200 dark:border-gray-700 px-6 py-4 flex-shrink-0">
                    <div class="flex justify-end gap-3">
                        <button type="button" onclick="closeFinalOfferModal()"
                            class="px-6 py-2.5 text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg font-medium hover:bg-gray-50 dark:hover:bg-gray-600 transition-all duration-200">
                            <i class="fas fa-times mr-2"></i>Batal
                        </button>
                        <button type="button" onclick="submitFinalOffer()"
                            class="px-6 py-2.5 bg-gray-700 hover:bg-gray-800 dark:bg-gray-600 dark:hover:bg-gray-700 text-white rounded-lg font-medium transition-all duration-200">
                            <i class="fas fa-paper-plane mr-2"></i>Kirim
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <script>
            function openInquiryModalFromBtn(btn) {
                openInquiryModal(Number(btn.dataset.id));
            }

            function forwardToTeknisiFromBtn(btn) {
                forwardToTeknisi(Number(btn.dataset.id));
            }
            
            function forwardToTeknisiFromModal(id) {
                forwardToTeknisi(id);
            }

            function forwardToSupervisorFromBtn(btn) {
                forwardToSupervisor(Number(btn.dataset.id));
            }
            
            function forwardToSupervisorFromModal(id) {
                forwardToSupervisor(id);
            }
            
            function openRevisionModal(id) {
                document.getElementById('revision-inquiry-id').value = id;
                
                // Reset form
                document.getElementById('revision-note').value = '';
                
                const modal = document.getElementById('revisionModal');
                const modalContent = document.getElementById('revisionModalContent');
                
                if (modal && modalContent) {
                    // Reset transform
                    modalContent.style.transform = 'translateX(100%)';
                    modalContent.style.opacity = '0';
                    
                    // Show modal
                    modal.classList.remove('hidden');
                    modal.classList.add('flex', 'items-center', 'justify-center');
                    document.body.style.overflow = 'hidden';
                    
                    // Slide in animation
                    setTimeout(() => {
                        modalContent.style.transform = 'translateX(0)';
                        modalContent.style.opacity = '1';
                    }, 10);
                }
            }
            
            function closeRevisionModal() {
                const modal = document.getElementById('revisionModal');
                const modalContent = document.getElementById('revisionModalContent');
                
                if (modal && modalContent) {
                    // Slide out animation
                    modalContent.style.transform = 'translateX(100%)';
                    modalContent.style.opacity = '0';
                    
                    // Hide modal after animation
                    setTimeout(() => {
                        modal.classList.remove('flex', 'items-center', 'justify-center');
                        modal.classList.add('hidden');
                        modalContent.style.transform = '';
                        modalContent.style.opacity = '';
                        document.body.style.overflow = 'auto';
                    }, 300);
                }
            }
            
            function submitRevisionRequest() {
                const id = document.getElementById('revision-inquiry-id').value;
                const revisionNote = document.getElementById('revision-note').value.trim();
                
                if (!revisionNote) {
                    Swal.fire({
                        title: 'Error!',
                        text: 'Catatan revisi harus diisi!',
                        icon: 'error',
                        confirmButtonText: 'OK'
                    });
                    return;
                }
                
                Swal.fire({
                    title: 'Kirim Permintaan Revisi?',
                    html: '<p class="text-sm text-gray-600 dark:text-gray-400 mb-3">Penawaran akan dikembalikan ke teknisi untuk diperbaiki sesuai catatan revisi Anda.</p>',
                    icon: 'question',
                    showCancelButton: true,
                    confirmButtonText: '<i class="fas fa-check mr-2"></i>Ya, Kirim',
                    cancelButtonText: '<i class="fas fa-times mr-2"></i>Batal',
                    confirmButtonColor: '#eab308',
                    cancelButtonColor: '#6b7280',
                    reverseButtons: true,
                    customClass: {
                        popup: 'rounded-lg',
                        confirmButton: 'px-4 py-2 rounded-md font-medium',
                        cancelButton: 'px-4 py-2 rounded-md font-medium'
                    }
                }).then((res) => {
                    if (!res.isConfirmed) return;
                    
                    fetch(`/admin/product-inquiries/${id}/request-revision`, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                        },
                        body: JSON.stringify({
                            revision_note: revisionNote
                        })
                    }).then(r => r.json()).then(data => {
                        if (data.success) {
                            closeRevisionModal();
                            Swal.fire('Berhasil', 'Permintaan revisi telah dikirim ke teknisi', 'success');
                            setTimeout(() => location.reload(), 1000);
                        } else {
                            Swal.fire('Gagal', data.message || 'Tidak dapat mengirim permintaan revisi', 'error');
                        }
                    }).catch(() => Swal.fire('Error', 'Terjadi kesalahan', 'error'));
                });
            }

            function markAsReadFromBtn(btn) {
                markAsRead(Number(btn.dataset.id));
            }

            function openConvertOrderModalFromBtn(btn) {
                openConvertOrderModal(
                    Number(btn.dataset.id),
                    btn.dataset.name,
                    btn.dataset.email || '',
                    btn.dataset.product,
                    Number(btn.dataset.price)
                );
            }

            function openFinalOfferModalFromData(btn) {
                try {
                    const payload = {
                        id: Number(btn.dataset.id),
                        productTitle: btn.dataset.title,
                        floor: Number(btn.dataset.floor),
                        target: Number(btn.dataset.target),
                        ceiling: Number(btn.dataset.ceiling),
                        validity: btn.dataset.validity || '',
                        internalCost: Number(btn.dataset.internalcost),
                        marginPercent: btn.dataset.margin !== '' ? Number(btn.dataset.margin) : null,
                    };
                    openFinalOfferModal(
                        payload.id,
                        payload.productTitle,
                        payload.floor,
                        payload.target,
                        payload.ceiling,
                        payload.validity,
                        payload.internalCost,
                        payload.marginPercent
                    );
                } catch (e) {
                    console.error('FinalOffer payload parse error:', e);
                    if (typeof Swal !== 'undefined' && Swal.fire) {
                        Swal.fire('Error', 'Data penawaran tidak valid', 'error');
                    }
                }
            }

            function openFinalOfferModal(id, productTitle, floor, target, ceiling, validity, internalCost, marginPercent) {
                document.getElementById('final-inquiry-id').value = id;
                document.getElementById('final-product-title').value = productTitle;
                document.getElementById('final-floor').innerText = floor ? 'Rp ' + new Intl.NumberFormat('id-ID').format(
                    floor) : '-';
                document.getElementById('final-target').innerText = target ? 'Rp ' + new Intl.NumberFormat('id-ID').format(
                    target) : '-';
                document.getElementById('final-ceiling').innerText = ceiling ? 'Rp ' + new Intl.NumberFormat('id-ID').format(
                    ceiling) : '-';
                document.getElementById('final-validity').innerText = validity || '-';
                const internalEl = document.getElementById('final-internal');
                const marginEl = document.getElementById('final-margin');
                if (internalEl) internalEl.innerText = (internalCost && Number(internalCost) > 0) ? ('Rp ' + new Intl
                    .NumberFormat('id-ID').format(internalCost)) : '-';
                if (marginEl) marginEl.innerText = (typeof marginPercent === 'number') ? ((marginPercent * 100).toFixed(1) +
                    '%') : '-';

                const modal = document.getElementById('finalOfferModal');
                const modalContent = document.getElementById('finalOfferModalContent');

                if (modal && modalContent) {
                    // Reset transform
                    modalContent.style.transform = 'translateX(100%)';
                    modalContent.style.opacity = '0';

                    // Show modal
                    modal.classList.remove('hidden');
                    modal.classList.add('flex', 'items-center', 'justify-center');
                    document.body.style.overflow = 'hidden';

                    // Slide in animation
                    setTimeout(() => {
                        modalContent.style.transform = 'translateX(0)';
                        modalContent.style.opacity = '1';
                    }, 10);
                }
            }

            function closeFinalOfferModal() {
                const modal = document.getElementById('finalOfferModal');
                const modalContent = document.getElementById('finalOfferModalContent');

                if (modal && modalContent) {
                    // Slide out animation
                    modalContent.style.transform = 'translateX(100%)';
                    modalContent.style.opacity = '0';

                    // Hide modal after animation
                    setTimeout(() => {
                        modal.classList.remove('flex', 'items-center', 'justify-center');
                        modal.classList.add('hidden');
                        modalContent.style.transform = '';
                        modalContent.style.opacity = '';
                        document.body.style.overflow = 'auto';
                    }, 300);
                }
            }

            function submitFinalOffer() {
                const id = document.getElementById('final-inquiry-id').value;
                const price = document.getElementById('final-price').value;
                const notes = document.getElementById('final-notes').value;
                fetch(`/admin/product-inquiries/${id}/final-approval`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    },
                    body: JSON.stringify({
                        final_price: price,
                        notes: notes
                    })
                }).then(r => r.json()).then(data => {
                    if (data.success) {
                        closeFinalOfferModal();
                        // Tawarkan kirim via WhatsApp
                        if (data.whatsapp_link) {
                            Swal.fire({
                                title: 'Berhasil',
                                html: 'Penawaran dikirim ke customer. Ingin kirim via WhatsApp juga?',
                                icon: 'success',
                                showCancelButton: true,
                                confirmButtonText: 'Buka WhatsApp',
                                cancelButtonText: 'Tutup'
                            }).then((res) => {
                                if (res.isConfirmed) {
                                    window.open(data.whatsapp_link, '_blank');
                                }
                                setTimeout(() => location.reload(), 800);
                            });
                            return;
                        }
                        Swal.fire('Berhasil', 'Penawaran dikirim ke customer', 'success');
                        setTimeout(() => location.reload(), 1000);
                    } else {
                        Swal.fire('Gagal', data.message || 'Gagal mengirim', 'error');
                    }
                }).catch(() => Swal.fire('Error', 'Terjadi kesalahan', 'error'));
            }

            function forwardToTeknisi(id) {
                Swal.fire({
                    title: 'Teruskan ke Teknisi?',
                    html: '<p class="text-sm text-gray-600 dark:text-gray-400 mb-3">Penawaran akan dikirim ke teknisi untuk melakukan estimasi biaya produksi.</p><p class="text-xs text-gray-500 dark:text-gray-500"><i class="fas fa-info-circle mr-1"></i>Teknisi akan menerima notifikasi dan dapat memberikan estimasi harga.</p>',
                    icon: 'question',
                    showCancelButton: true,
                    confirmButtonText: '<i class="fas fa-check mr-2"></i>Ya, Teruskan',
                    cancelButtonText: '<i class="fas fa-times mr-2"></i>Batal',
                    confirmButtonColor: '#10b981',
                    cancelButtonColor: '#6b7280',
                    reverseButtons: true,
                    customClass: {
                        popup: 'rounded-lg',
                        confirmButton: 'px-4 py-2 rounded-md font-medium',
                        cancelButton: 'px-4 py-2 rounded-md font-medium'
                    }
                }).then((res) => {
                    if (!res.isConfirmed) return;
                    fetch(`/admin/product-inquiries/${id}/forward-teknisi`, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute(
                                'content')
                        }
                    }).then(r => r.json()).then(data => {
                        if (data.success) {
                            Swal.fire('Berhasil', 'Diteruskan ke Teknisi', 'success');
                            setTimeout(() => location.reload(), 800);
                        } else {
                            Swal.fire('Gagal', data.message || 'Tidak dapat meneruskan', 'error');
                        }
                    }).catch(() => Swal.fire('Error', 'Terjadi kesalahan', 'error'));
                });
            }

            function forwardToSupervisor(id) {
                Swal.fire({
                    title: 'Teruskan ke Supervisor?',
                    html: '<p class="text-sm text-gray-600 dark:text-gray-400 mb-3">Penawaran akan dikirim ke supervisor untuk menetapkan koridor harga (floor, target, ceiling).</p><p class="text-xs text-gray-500 dark:text-gray-500"><i class="fas fa-info-circle mr-1"></i>Supervisor akan menerima notifikasi dan dapat menentukan rentang harga yang sesuai.</p>',
                    icon: 'question',
                    showCancelButton: true,
                    confirmButtonText: '<i class="fas fa-check mr-2"></i>Ya, Teruskan',
                    cancelButtonText: '<i class="fas fa-times mr-2"></i>Batal',
                    confirmButtonColor: '#10b981',
                    cancelButtonColor: '#6b7280',
                    reverseButtons: true,
                    customClass: {
                        popup: 'rounded-lg',
                        confirmButton: 'px-4 py-2 rounded-md font-medium',
                        cancelButton: 'px-4 py-2 rounded-md font-medium'
                    }
                }).then((res) => {
                    if (!res.isConfirmed) return;
                    fetch(`/admin/product-inquiries/${id}/forward-supervisor`, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute(
                                'content')
                        }
                    }).then(r => r.json()).then(data => {
                        if (data.success) {
                            Swal.fire('Berhasil', 'Diteruskan ke Supervisor', 'success');
                            setTimeout(() => location.reload(), 800);
                        } else {
                            Swal.fire('Gagal', data.message || 'Tidak dapat meneruskan', 'error');
                        }
                    }).catch(() => Swal.fire('Error', 'Terjadi kesalahan', 'error'));
                });
            }
            
            function openRevisionModal(id) {
                document.getElementById('revision-inquiry-id').value = id;
                
                // Reset form
                document.getElementById('revision-note').value = '';
                
                const modal = document.getElementById('revisionModal');
                const modalContent = document.getElementById('revisionModalContent');
                
                if (modal && modalContent) {
                    // Reset transform
                    modalContent.style.transform = 'translateX(100%)';
                    modalContent.style.opacity = '0';
                    
                    // Show modal
                    modal.classList.remove('hidden');
                    modal.classList.add('flex', 'items-center', 'justify-center');
                    document.body.style.overflow = 'hidden';
                    
                    // Slide in animation
                    setTimeout(() => {
                        modalContent.style.transform = 'translateX(0)';
                        modalContent.style.opacity = '1';
                    }, 10);
                }
            }
            
            function closeRevisionModal() {
                const modal = document.getElementById('revisionModal');
                const modalContent = document.getElementById('revisionModalContent');
                
                if (modal && modalContent) {
                    // Slide out animation
                    modalContent.style.transform = 'translateX(100%)';
                    modalContent.style.opacity = '0';
                    
                    // Hide modal after animation
                    setTimeout(() => {
                        modal.classList.remove('flex', 'items-center', 'justify-center');
                        modal.classList.add('hidden');
                        modalContent.style.transform = '';
                        modalContent.style.opacity = '';
                        document.body.style.overflow = 'auto';
                    }, 300);
                }
            }
            
            function submitRevisionRequest() {
                const id = document.getElementById('revision-inquiry-id').value;
                const revisionNote = document.getElementById('revision-note').value.trim();
                
                if (!revisionNote) {
                    Swal.fire({
                        title: 'Error!',
                        text: 'Catatan revisi harus diisi!',
                        icon: 'error',
                        confirmButtonText: 'OK'
                    });
                    return;
                }
                
                Swal.fire({
                    title: 'Kirim Permintaan Revisi?',
                    html: '<p class="text-sm text-gray-600 dark:text-gray-400 mb-3">Penawaran akan dikembalikan ke teknisi untuk diperbaiki sesuai catatan revisi Anda.</p>',
                    icon: 'question',
                    showCancelButton: true,
                    confirmButtonText: '<i class="fas fa-check mr-2"></i>Ya, Kirim',
                    cancelButtonText: '<i class="fas fa-times mr-2"></i>Batal',
                    confirmButtonColor: '#eab308',
                    cancelButtonColor: '#6b7280',
                    reverseButtons: true,
                    customClass: {
                        popup: 'rounded-lg',
                        confirmButton: 'px-4 py-2 rounded-md font-medium',
                        cancelButton: 'px-4 py-2 rounded-md font-medium'
                    }
                }).then((res) => {
                    if (!res.isConfirmed) return;
                    
                    fetch(`/admin/product-inquiries/${id}/request-revision`, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                        },
                        body: JSON.stringify({
                            revision_note: revisionNote
                        })
                    }).then(r => r.json()).then(data => {
                        if (data.success) {
                            closeRevisionModal();
                            Swal.fire('Berhasil', 'Permintaan revisi telah dikirim ke teknisi', 'success');
                            setTimeout(() => location.reload(), 1000);
                        } else {
                            Swal.fire('Gagal', data.message || 'Tidak dapat mengirim permintaan revisi', 'error');
                        }
                    }).catch(() => Swal.fire('Error', 'Terjadi kesalahan', 'error'));
                });
            }

            // Copy Link Dokumen Penawaran
            function copyPreviewLinkFromBtn(btn) {
                const url = btn.dataset.previewUrl || '';
                if (!url) {
                    Swal.fire('Info', 'Link dokumen tidak tersedia', 'info');
                    return;
                }
                if (navigator.clipboard && navigator.clipboard.writeText) {
                    navigator.clipboard.writeText(url).then(() => {
                        Swal.fire('Tersalin', 'Link dokumen berhasil disalin', 'success');
                    }).catch(() => {
                        Swal.fire('Gagal', 'Tidak dapat menyalin ke clipboard', 'error');
                    });
                } else {
                    // Fallback untuk browser lama
                    const ta = document.createElement('textarea');
                    ta.value = url;
                    document.body.appendChild(ta);
                    ta.select();
                    try {
                        document.execCommand('copy');
                        Swal.fire('Tersalin', 'Link dokumen berhasil disalin', 'success');
                    } catch (e) {
                        Swal.fire('Gagal', 'Tidak dapat menyalin ke clipboard', 'error');
                    }
                    document.body.removeChild(ta);
                }
            }

            // Convert to Order Modal Functions
            function openConvertOrderModal(inquiryId, customerName, customerEmail, productTitle, finalPrice) {
                document.getElementById('convert-inquiry-id').value = inquiryId;
                document.getElementById('convert-customer-name').textContent = customerName;
                document.getElementById('convert-product-title').textContent = productTitle;
                document.getElementById('convert-final-price').textContent = 'Rp ' + new Intl.NumberFormat('id-ID').format(
                    finalPrice);
                document.getElementById('convert-order-price').value = finalPrice;

                // Simpan email untuk digunakan di toggleAddressInput
                currentInquiryEmail = customerEmail || '';

                // Reset form
                document.getElementById('useDifferentAddress').checked = false;
                document.getElementById('convert-address').value = '';
                document.getElementById('convert-address').disabled = false;

                // Fetch alamat user jika email tersedia
                if (customerEmail) {
                    fetchUserAddress(customerEmail);
                } else {
                    // Jika tidak ada email, enable textarea untuk diisi manual
                    document.getElementById('useDifferentAddress').checked = true;
                    document.getElementById('convert-address').placeholder = 'Masukkan alamat lengkap customer';
                }

                const modal = document.getElementById('convertOrderModal');
                const modalContent = document.getElementById('convertOrderModalContent');

                if (modal && modalContent) {
                    // Reset transform
                    modalContent.style.transform = 'translateX(100%)';
                    modalContent.style.opacity = '0';

                    // Show modal
                    modal.classList.remove('hidden');
                    modal.classList.add('flex', 'items-center', 'justify-center');
                    document.body.style.overflow = 'hidden';

                    // Slide in animation
                    setTimeout(() => {
                        modalContent.style.transform = 'translateX(0)';
                        modalContent.style.opacity = '1';
                    }, 10);
                }
            }

            // Fetch alamat user berdasarkan email
            function fetchUserAddress(email) {
                const textarea = document.getElementById('convert-address');
                const checkbox = document.getElementById('useDifferentAddress');

                // Show loading state
                textarea.disabled = true;
                textarea.value = 'Memuat alamat dari profil user...';

                fetch(`/admin/product-inquiries/get-user-address?email=${encodeURIComponent(email)}`, {
                        method: 'GET',
                        headers: {
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                            'Accept': 'application/json'
                        }
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success && data.address) {
                            // Alamat ditemukan, isi ke textarea
                            textarea.value = data.address;
                            textarea.disabled = false;
                            checkbox.checked = false;
                            textarea.placeholder = 'Alamat dari profil user. Centang "Gunakan alamat lain" untuk mengubah';
                        } else {
                            // Jika tidak ada alamat di profil, enable textarea untuk diisi manual
                            textarea.value = '';
                            textarea.disabled = false;
                            checkbox.checked = true;
                            textarea.placeholder = 'Masukkan alamat lengkap customer';
                            textarea.focus();
                        }
                    })
                    .catch(error => {
                        console.error('Error fetching user address:', error);
                        // Jika error, enable textarea untuk diisi manual
                        textarea.value = '';
                        textarea.disabled = false;
                        checkbox.checked = true;
                        textarea.placeholder = 'Masukkan alamat lengkap customer';
                        textarea.focus();
                    });
            }

            function closeConvertOrderModal() {
                const modal = document.getElementById('convertOrderModal');
                const modalContent = document.getElementById('convertOrderModalContent');

                if (modal && modalContent) {
                    // Slide out animation
                    modalContent.style.transform = 'translateX(100%)';
                    modalContent.style.opacity = '0';

                    // Hide modal after animation
                    setTimeout(() => {
                        modal.classList.remove('flex', 'items-center', 'justify-center');
                        modal.classList.add('hidden');
                        modalContent.style.transform = '';
                        modalContent.style.opacity = '';
                        document.body.style.overflow = 'auto';
                    }, 300);
                }

                // Reset variabel
                currentInquiryEmail = null;
            }

            // Toggle address input berdasarkan checkbox
            function toggleAddressInput() {
                const checkbox = document.getElementById('useDifferentAddress');
                const textarea = document.getElementById('convert-address');

                if (checkbox.checked) {
                    // Jika dicentang, enable dan focus ke textarea untuk input manual
                    textarea.disabled = false;
                    textarea.focus();
                    if (!textarea.value) {
                        textarea.placeholder = 'Masukkan alamat lengkap customer';
                    }
                } else {
                    // Jika tidak dicentang, fetch alamat user lagi dari profil
                    if (currentInquiryEmail) {
                        fetchUserAddress(currentInquiryEmail);
                    } else {
                        textarea.value = '';
                        textarea.placeholder =
                            'Alamat akan diambil dari profil user, centang "Gunakan alamat lain" untuk mengubah';
                    }
                }
            }

            function submitConvertOrder() {
                const form = document.getElementById('convertOrderForm');
                const formData = new FormData(form);
                const inquiryId = formData.get('inquiry_id');

                fetch(`/admin/product-inquiries/${inquiryId}/convert-to-order`, {
                        method: 'POST',
                        headers: {
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                            'Accept': 'application/json'
                        },
                        body: formData
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            closeConvertOrderModal();
                            Swal.fire({
                                title: 'Berhasil!',
                                html: `Inquiry berhasil di-convert ke Order #${data.order_id}.<br><small>${data.message}</small>`,
                                icon: 'success',
                                showCancelButton: true,
                                confirmButtonText: 'Lihat Order',
                                cancelButtonText: 'Tutup'
                            }).then((result) => {
                                if (result.isConfirmed && data.redirect_url) {
                                    window.location.href = data.redirect_url;
                                } else {
                                    setTimeout(() => location.reload(), 1000);
                                }
                            });
                        } else {
                            Swal.fire({
                                title: 'Gagal!',
                                text: data.message || 'Gagal convert ke order',
                                icon: 'error',
                                confirmButtonText: 'OK'
                            });
                        }
                    })
                    .catch(error => {
                        console.error('Error:', error);
                        Swal.fire({
                            title: 'Error!',
                            text: 'Terjadi kesalahan saat memproses',
                            icon: 'error',
                            confirmButtonText: 'OK'
                        });
                    });
            }
        </script>

        <!-- Convert to Order Modal -->
        <div id="convertOrderModal"
            class="fixed inset-0 bg-black/60 backdrop-blur-sm hidden items-center justify-center z-[9999] transition-all duration-300 ease-in-out">
            <div id="convertOrderModalContent"
                class="bg-white dark:bg-gray-800 rounded-lg border border-gray-300 dark:border-gray-600 shadow-2xl max-w-2xl w-full mx-4 p-0 overflow-hidden transform transition-all duration-300 ease-in-out max-h-[90vh] overflow-y-auto">
                <!-- Header -->
                <div class="bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700 px-6 py-4">
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-3">
                            <div class="w-10 h-10 bg-gray-100 dark:bg-gray-700 rounded-lg flex items-center justify-center">
                                <i class="fas fa-shopping-cart text-gray-600 dark:text-gray-300 text-lg"></i>
                            </div>
                            <h2 class="text-xl font-semibold text-gray-900 dark:text-white">Buat Pesanan</h2>
                        </div>
                        <button onclick="closeConvertOrderModal()"
                            class="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 transition-colors duration-200 p-1 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700">
                            <i class="fas fa-times text-lg"></i>
                        </button>
                    </div>
                </div>

                <!-- Content -->
                <div class="p-6">

                    <form id="convertOrderForm" onsubmit="event.preventDefault(); submitConvertOrder();">
                        <input type="hidden" name="inquiry_id" id="convert-inquiry-id" value="">

                        <!-- Customer Info Card -->
                        <div
                            class="bg-gray-50 dark:bg-gray-700/30 rounded-lg border border-gray-200 dark:border-gray-600 p-4 mb-4">
                            <div class="flex items-center mb-2">
                                <div class="p-2 bg-gray-100 dark:bg-gray-600 rounded-lg mr-2">
                                    <i class="fas fa-user text-gray-600 dark:text-gray-400 text-sm"></i>
                                </div>
                                <h3 class="text-sm font-semibold text-gray-800 dark:text-white">Informasi Customer</h3>
                            </div>
                            <p class="text-sm text-gray-900 dark:text-gray-100 font-medium" id="convert-customer-name"></p>
                        </div>

                        <!-- Product Info Card -->
                        <div
                            class="bg-gray-50 dark:bg-gray-700/30 rounded-lg border border-gray-200 dark:border-gray-600 p-4 mb-4">
                            <div class="flex items-center mb-2">
                                <div class="p-2 bg-gray-100 dark:bg-gray-600 rounded-lg mr-2">
                                    <i class="fas fa-box text-gray-600 dark:text-gray-400 text-sm"></i>
                                </div>
                                <h3 class="text-sm font-semibold text-gray-800 dark:text-white">Informasi Produk</h3>
                            </div>
                            <p class="text-sm text-gray-900 dark:text-gray-100 font-medium mb-2" id="convert-product-title">
                            </p>
                            <div class="flex items-center">
                                <span class="text-xs text-gray-600 dark:text-gray-400 mr-2">Harga Final:</span>
                                <span class="text-sm font-semibold text-gray-700 dark:text-gray-300"
                                    id="convert-final-price"></span>
                            </div>
                        </div>

                        <!-- Order Details -->
                        <div class="space-y-4">
                            <div>
                                <label class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                    <i class="fas fa-boxes mr-2 text-gray-500"></i>Jumlah <span class="text-red-500">*</span>
                                </label>
                                <input type="number" name="quantity" value="1" min="1" required
                                    class="w-full px-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white transition-colors duration-200">
                            </div>

                            <div>
                                <div class="flex items-center justify-between mb-2">
                                    <label class="block text-sm font-semibold text-gray-700 dark:text-gray-300">
                                        <i class="fas fa-map-marker-alt mr-2 text-gray-500"></i>Alamat Lengkap <span
                                            class="text-red-500">*</span>
                                    </label>
                                    <label class="flex items-center gap-2 cursor-pointer">
                                        <input type="checkbox" id="useDifferentAddress" onchange="toggleAddressInput()"
                                            class="w-4 h-4 text-blue-600 rounded focus:ring-blue-500">
                                        <span class="text-xs text-gray-600 dark:text-gray-400">Gunakan alamat lain</span>
                                    </label>
                                </div>
                                <textarea name="address" id="convert-address" rows="3" required
                                    class="w-full px-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white resize-none transition-colors duration-200"
                                    placeholder="Alamat akan diambil dari profil user, centang 'Gunakan alamat lain' untuk mengubah"></textarea>
                                <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                    <i class="fas fa-info-circle mr-1"></i>Alamat akan digunakan untuk pengiriman
                                </p>
                            </div>

                            <div>
                                <label class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                    <i class="fas fa-credit-card mr-2 text-gray-500"></i>Metode Pembayaran <span
                                        class="text-red-500">*</span>
                                </label>
                                <select name="payment_method" required
                                    class="w-full px-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white transition-colors duration-200">
                                    <option value="transfer">Transfer Bank</option>
                                    <option value="midtrans">Midtrans (Online Payment)</option>
                                    <option value="dp">Down Payment (DP)</option>
                                </select>
                            </div>

                            <div>
                                <label class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                    <i class="fas fa-tasks mr-2 text-gray-500"></i>Status Pesanan <span
                                        class="text-red-500">*</span>
                                </label>
                                <select name="order_status" required
                                    class="w-full px-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white transition-colors duration-200">
                                    <option value="menunggu">Menunggu</option>
                                    <option value="diterima">Diterima (Mulai Produksi)</option>
                                    <option value="ditolak">Ditolak</option>
                                </select>
                                <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                    <i class="fas fa-info-circle mr-1"></i>Jika "Diterima", produksi akan dibuat otomatis
                                </p>
                            </div>

                            <input type="hidden" name="price" id="convert-order-price" value="">
                        </div>

                        <!-- Warning -->
                        <div
                            class="bg-gray-50 dark:bg-gray-700/30 border border-gray-200 dark:border-gray-600 rounded-lg p-4 mt-4">
                            <div class="flex items-start">
                                <i class="fas fa-exclamation-triangle text-gray-600 dark:text-gray-400 mr-2 mt-0.5"></i>
                                <div>
                                    <p class="text-sm font-semibold text-gray-800 dark:text-white mb-1">Peringatan</p>
                                    <p class="text-xs text-gray-600 dark:text-gray-400">
                                        Pastikan customer sudah benar-benar setuju dan siap untuk memesan.
                                        Setelah di-convert, penawaran akan ditandai sebagai "Jadi Pesanan".
                                    </p>
                                </div>
                            </div>
                        </div>

                        <!-- Actions -->
                        <div class="flex justify-end gap-3 mt-6 pt-4 border-t border-gray-200 dark:border-gray-700">
                            <button type="button" onclick="closeConvertOrderModal()"
                                class="px-6 py-2.5 text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg font-medium hover:bg-gray-50 dark:hover:bg-gray-600 transition-all duration-200">
                                <i class="fas fa-times mr-2"></i>Batal
                            </button>
                            <button type="submit"
                                class="px-6 py-2.5 bg-gray-700 hover:bg-gray-800 dark:bg-gray-600 dark:hover:bg-gray-700 text-white rounded-lg font-medium transition-all duration-200 flex items-center">
                                <i class="fas fa-shopping-cart mr-2"></i>
                                Buat Pesanan
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>


        <style>
            /* Custom scrollbar untuk tabel */
            .overflow-x-auto::-webkit-scrollbar {
                height: 8px;
            }

            .overflow-x-auto::-webkit-scrollbar-track {
                background: #f1f5f9;
                border-radius: 4px;
            }

            .overflow-x-auto::-webkit-scrollbar-thumb {
                background: #cbd5e0;
                border-radius: 4px;
            }

            .overflow-x-auto::-webkit-scrollbar-thumb:hover {
                background: #94a3b8;
            }

            /* Dark mode scrollbar */
            .dark .overflow-x-auto::-webkit-scrollbar-track {
                background: #374151;
            }

            .dark .overflow-x-auto::-webkit-scrollbar-thumb {
                background: #6b7280;
            }

            .dark .overflow-x-auto::-webkit-scrollbar-thumb:hover {
                background: #9ca3af;
            }

            /* Modal Animation */
            #inquiryModalContent,
            #replyModalContent,
            #finalOfferModalContent,
            #convertOrderModalContent,
            #revisionModalContent {
                transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
            }
        </style>

        {{-- Modal Detail Pesan --}}
        <div id="inquiryModal"
            class="fixed inset-0 bg-black/60 backdrop-blur-sm hidden items-center justify-center z-[9999] transition-all duration-300 ease-in-out">
            <div id="inquiryModalContent"
                class="bg-white dark:bg-gray-800 rounded-lg border border-gray-300 dark:border-gray-600 shadow-2xl max-w-3xl w-full mx-4 p-0 overflow-hidden transform transition-all duration-300 ease-in-out max-h-[90vh] overflow-y-auto">
                <!-- Header -->
                <div class="bg-gradient-to-r from-gray-700 to-gray-800 dark:from-gray-800 dark:to-gray-900 px-6 py-5 shadow-lg border-b border-gray-600 dark:border-gray-700">
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-4">
                            <div class="w-12 h-12 bg-gray-600 dark:bg-gray-700 rounded-lg flex items-center justify-center shadow-md">
                                <i class="fas fa-file-alt text-white text-xl"></i>
                            </div>
                            <div>
                                <h2 class="text-2xl font-bold text-white">Detail Penawaran</h2>
                                <p class="text-sm text-gray-300 dark:text-gray-400 mt-0.5">Informasi lengkap penawaran produk</p>
                            </div>
                        </div>
                        <button onclick="closeInquiryModal()"
                            class="text-gray-300 hover:text-white transition-all duration-200 p-2 rounded-lg hover:bg-gray-600 dark:hover:bg-gray-700">
                            <i class="fas fa-times text-xl"></i>
                        </button>
                    </div>
                </div>

                <!-- Content -->
                <div class="p-6">
                    <div id="inquiryDetails" class="space-y-4 mb-6">
                        <!-- Konten akan diisi via JS -->
                    </div>

                    <!-- Footer Actions -->
                    <div class="flex justify-end space-x-3 pt-5 border-t border-gray-200 dark:border-gray-700 bg-gray-50 dark:bg-gray-800/50">
                        <button type="button" onclick="closeInquiryModal()"
                            class="px-6 py-3 text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg font-semibold hover:bg-gray-50 dark:hover:bg-gray-600 transition-all duration-200 shadow-sm hover:shadow">
                            <i class="fas fa-times mr-2"></i>Tutup
                        </button>
                        <button id="markAsReadBtn" onclick="markAsReadFromModal()"
                            class="px-6 py-3 text-white bg-blue-600 hover:bg-blue-700 rounded-lg font-semibold transition-all duration-200 shadow-sm hover:shadow">
                            <i class="fas fa-check mr-2"></i>Tandai Dibaca
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Revision Request Modal -->
        <div id="revisionModal"
            class="fixed inset-0 bg-black/60 backdrop-blur-sm hidden items-center justify-center z-[9999] transition-all duration-300 ease-in-out">
            <div id="revisionModalContent"
                class="bg-white dark:bg-gray-800 rounded-lg border border-gray-300 dark:border-gray-600 shadow-2xl max-w-2xl w-full mx-4 p-0 overflow-hidden transform transition-all duration-300 ease-in-out max-h-[90vh] overflow-y-auto">
                <!-- Header -->
                <div class="bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700 px-6 py-4">
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-3">
                            <div class="w-10 h-10 bg-yellow-100 dark:bg-yellow-900/40 rounded-lg flex items-center justify-center">
                                <i class="fas fa-edit text-yellow-600 dark:text-yellow-400 text-lg"></i>
                            </div>
                            <h2 class="text-xl font-semibold text-gray-900 dark:text-white">Minta Revisi Estimasi</h2>
                        </div>
                        <button onclick="closeRevisionModal()"
                            class="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 transition-colors duration-200 p-1 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700">
                            <i class="fas fa-times text-lg"></i>
                        </button>
                    </div>
                </div>

                <!-- Content -->
                <div class="p-6">
                    <form id="revisionForm">
                        @csrf
                        <input type="hidden" id="revision-inquiry-id" />
                        <div class="space-y-4">
                            <!-- Warning Info -->
                            <div class="bg-yellow-50 dark:bg-yellow-900/20 rounded-lg border border-yellow-200 dark:border-yellow-700 p-4">
                                <div class="flex items-start">
                                    <i class="fas fa-info-circle text-yellow-600 dark:text-yellow-400 mt-1 mr-3"></i>
                                    <div>
                                        <p class="text-sm font-semibold text-yellow-900 dark:text-yellow-100 mb-1">
                                            Permintaan Revisi
                                        </p>
                                        <p class="text-xs text-yellow-700 dark:text-yellow-300">
                                            Tuliskan catatan revisi yang jelas untuk teknisi. Penawaran akan dikembalikan ke teknisi untuk diperbaiki.
                                        </p>
                                    </div>
                                </div>
                            </div>

                            <!-- Revision Note -->
                            <div>
                                <label class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                    <i class="fas fa-comment-dots mr-2 text-gray-500"></i>Catatan Revisi <span class="text-red-500">*</span>
                                </label>
                                <textarea id="revision-note" name="revision_note" rows="6" required
                                    class="w-full px-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-yellow-500 resize-none dark:bg-gray-700 dark:text-white transition-colors duration-200"
                                    placeholder="Contoh: Material breakdown kurang detail, perlu tambahkan spesifikasi lengkap. Labor cost perlu disesuaikan dengan kompleksitas pekerjaan..."></textarea>
                                <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                    Jelaskan bagian mana yang perlu direvisi dan alasan revisi
                                </p>
                            </div>
                        </div>
                    </form>
                </div>

                <!-- Footer Actions -->
                <div class="bg-white dark:bg-gray-800 border-t border-gray-200 dark:border-gray-700 px-6 py-4">
                    <div class="flex justify-end gap-3">
                        <button type="button" onclick="closeRevisionModal()"
                            class="px-6 py-2.5 text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg font-medium hover:bg-gray-50 dark:hover:bg-gray-600 transition-all duration-200">
                            <i class="fas fa-times mr-2"></i>Batal
                        </button>
                        <button type="button" onclick="submitRevisionRequest()"
                            class="px-6 py-2.5 bg-yellow-600 hover:bg-yellow-700 text-white rounded-lg font-medium transition-all duration-200">
                            <i class="fas fa-paper-plane mr-2"></i>Kirim Permintaan Revisi
                        </button>
                    </div>
                </div>
            </div>
        </div>

        {{-- Modal Reply --}}
        <div id="replyModal"
            class="fixed inset-0 bg-black/60 backdrop-blur-sm hidden items-center justify-center z-[9999] transition-all duration-300 ease-in-out">
            <div id="replyModalContent"
                class="bg-white dark:bg-gray-800 rounded-lg border border-gray-300 dark:border-gray-600 shadow-2xl max-w-2xl w-full mx-4 p-0 overflow-hidden transform transition-all duration-300 ease-in-out max-h-[90vh] overflow-y-auto">
                <!-- Header -->
                <div class="bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700 px-6 py-4">
                    <div class="flex justify-between items-center">
                        <div class="flex items-center space-x-3">
                            <div class="w-10 h-10 bg-gray-100 dark:bg-gray-700 rounded-lg flex items-center justify-center">
                                <i class="fas fa-envelope text-gray-600 dark:text-gray-300 text-lg"></i>
                            </div>
                            <h2 class="text-xl font-semibold text-gray-900 dark:text-white">Balas Penawaran</h2>
                        </div>
                        <button onclick="closeReplyModal()"
                            class="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 transition-colors duration-200 p-1 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700">
                            <i class="fas fa-times text-lg"></i>
                        </button>
                    </div>
                </div>

                <!-- Content -->
                <div class="p-6">
                    <div id="replyForm" class="space-y-4 mb-6">
                        @csrf
                        <input type="hidden" id="reply-inquiry-id" value="" />

                        <!-- Customer Info Card -->
                        <div class="bg-gray-50 dark:bg-gray-700/30 rounded-lg border border-gray-200 dark:border-gray-600 p-4">
                            <div class="flex items-center mb-2">
                                <div class="p-2 bg-gray-100 dark:bg-gray-600 rounded-lg mr-2">
                                    <i class="fas fa-user text-gray-600 dark:text-gray-400 text-sm"></i>
                                </div>
                                <h3 class="text-sm font-semibold text-gray-800 dark:text-white">Kepada</h3>
                            </div>
                            <p id="replyTo" class="text-sm text-gray-900 dark:text-gray-100 font-medium"></p>
                        </div>

                        <!-- Metode Reply -->
                        <div>
                            <label for="replyMethod"
                                class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                <i class="fas fa-paper-plane mr-2 text-gray-500"></i>Metode Balasan
                            </label>
                            <select id="replyMethod" name="reply_method" required
                                class="w-full px-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 bg-white dark:bg-gray-700 text-gray-900 dark:text-white transition-colors duration-200">
                                <option value="">Pilih Metode Balasan</option>
                                <option value="email">📧 Email</option>
                                <option value="whatsapp">📱 WhatsApp</option>
                                <option value="phone">📞 Telepon</option>
                            </select>
                            <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Pilih metode yang akan digunakan untuk
                                mengirim balasan</p>
                        </div>

                        <!-- Subject -->
                        <div>
                            <label for="replySubject"
                                class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                <i class="fas fa-tag mr-2 text-gray-500"></i>Subjek
                            </label>
                            <input type="text" id="replySubject" name="subject"
                                class="w-full px-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 bg-white dark:bg-gray-700 text-gray-900 dark:text-white transition-colors duration-200"
                                placeholder="Contoh: Re: Penawaran Produk" required>
                        </div>

                        <!-- Message -->
                        <div>
                            <label for="replyMessage"
                                class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                <i class="fas fa-comment-dots mr-2 text-gray-500"></i>Pesan Balasan
                            </label>
                            <textarea id="replyMessage" name="message" rows="6"
                                class="w-full px-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 resize-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white transition-colors duration-200"
                                placeholder="Tulis pesan balasan Anda di sini..." required></textarea>
                            <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Tulis pesan balasan yang jelas dan
                                informatif</p>
                        </div>
                    </div>

                    <!-- Footer Actions -->
                    <div class="flex justify-end space-x-3 pt-4 border-t border-gray-200 dark:border-gray-700">
                        <!-- Tombol Batal dan Kirim Reply (default) -->
                        <div id="defaultButtons" class="flex space-x-3">
                            <button type="button" id="batalReplyBtn"
                                class="px-6 py-2.5 text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg font-medium hover:bg-gray-50 dark:hover:bg-gray-600 transition-all duration-200">
                                <i class="fas fa-times mr-2"></i>Batal
                            </button>
                            <button type="button" id="kirimReplyBtn"
                                class="px-6 py-2.5 text-white bg-gray-700 hover:bg-gray-800 dark:bg-gray-600 dark:hover:bg-gray-700 rounded-lg font-medium transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed">
                                <i class="fas fa-paper-plane mr-2" id="kirimReplyIcon"></i>
                                <span id="kirimReplyText">Kirim Balasan</span>
                                <span id="kirimReplyLoading" class="hidden">
                                    <i class="fas fa-spinner fa-spin mr-2"></i>Mengirim...
                                </span>
                            </button>
                        </div>

                        <!-- Tombol Selesai (hidden by default, muncul saat proses) -->
                        <button type="button" id="selesaiReplyBtn"
                            class="hidden px-6 py-2.5 text-white bg-gray-700 hover:bg-gray-800 dark:bg-gray-600 dark:hover:bg-gray-700 rounded-lg font-medium transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed">
                            <i class="fas fa-check mr-2"></i>Selesai
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <script>
            let currentInquiryId = null;
            let currentInquiryEmail = null;

            function openInquiryModal(id) {
                currentInquiryId = id;
                const row = document.querySelector(`tr[data-inquiry-id="${id}"]`);
                if (row) {
                    const name = row.dataset.inquiryName;
                    const email = row.dataset.inquiryEmail;
                    const phone = row.dataset.inquiryPhone;
                    const message = row.dataset.inquiryMessage;
                    const product = row.dataset.inquiryProduct;
                    const tracking = row.dataset.inquiryTracking;
                    const date = row.dataset.inquiryDate;
                    const isRead = parseInt(row.dataset.isRead);
                    const hasReply = parseInt(row.dataset.hasReply);

                    function parseInquiryMessage(msg) {
                        const typeMatch = msg.match(/\[REQUEST_TYPE\]\s*([^\r\n]+)/);
                        const addMatch = msg.match(/\[ADDONS\]\s*(\[[\s\S]*?\])/);
                        const type = typeMatch ? typeMatch[1].trim() : null;
                        let addons = [];
                        if (addMatch) {
                            try {
                                addons = JSON.parse(addMatch[1]);
                            } catch (e) {
                                addons = [];
                            }
                        }
                        let plain = msg.replace(/\[REQUEST_TYPE\][^\r\n]*/, '').replace(/\[ADDONS\][^\r\n]*/, '').trim();
                        return {
                            type,
                            addons,
                            plain
                        };
                    }

                    const parsed = parseInquiryMessage(message);
                    // Gunakan inquiry_type dari data attribute jika ada, jika tidak gunakan parsed type
                    const inquiryType = row.dataset.inquiryType || parsed.type || 'catalog';
                    const badgeText = inquiryType === 'custom_addon' ? 'Custom Add-on' : (inquiryType === 'custom_new' ?
                        'Custom Baru' : 'Katalog');
                    const badgeCls = inquiryType === 'custom_addon' ? 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/40 dark:text-yellow-200' : (inquiryType ===
                        'custom_new' ? 'bg-red-100 text-red-800 dark:bg-red-900/40 dark:text-red-200' : 'bg-blue-100 text-blue-800 dark:bg-blue-900/40 dark:text-blue-200');
                    
                    // Tentukan rekomendasi alur berdasarkan tipe inquiry
                    let workflowRecommendation = null;
                    if (inquiryType === 'catalog') {
                        workflowRecommendation = {
                            next_step: 'supervisor',
                            message: 'Produk katalog dapat langsung forward ke supervisor untuk set koridor harga.',
                            skip_teknisi: true
                        };
                    } else if (inquiryType === 'custom_addon') {
                        workflowRecommendation = {
                            next_step: 'teknisi',
                            message: 'Produk dengan spesifikasi tambahan perlu estimasi teknisi terlebih dahulu.',
                            skip_teknisi: false
                        };
                    } else if (inquiryType === 'custom_new') {
                        workflowRecommendation = {
                            next_step: 'teknisi',
                            message: 'Produk baru perlu teknisi merancang dan memberikan detail RAB lengkap.',
                            skip_teknisi: false
                        };
                    }
                    
                    // Ambil data estimasi teknisi
                    const status = row.dataset.inquiryStatus || '';
                    let materialsBreakdown = [];
                    let sparepartsBreakdown = [];
                    try {
                        materialsBreakdown = row.dataset.materialsBreakdown ? JSON.parse(row.dataset.materialsBreakdown) : [];
                    } catch (e) {
                        console.error('Error parsing materials breakdown:', e);
                        materialsBreakdown = [];
                    }
                    try {
                        sparepartsBreakdown = row.dataset.sparepartsBreakdown ? JSON.parse(row.dataset.sparepartsBreakdown) : [];
                    } catch (e) {
                        console.error('Error parsing spareparts breakdown:', e);
                        sparepartsBreakdown = [];
                    }
                    const laborCost = parseFloat(row.dataset.laborCost || 0);
                    const laborCostTotal = parseFloat(row.dataset.laborCostTotal || 0);
                    const materialsTotal = parseFloat(row.dataset.materialsTotal || 0);
                    const sparepartsTotal = parseFloat(row.dataset.sparepartsTotal || 0);
                    const totalCost = parseFloat(row.dataset.totalCost || 0);
                    const engineerSuggested = parseFloat(row.dataset.engineerSuggested || 0);
                    const estimatedDays = parseInt(row.dataset.estimatedDays || 0);
                    const teknisiNote = row.dataset.teknisiNote || '';
                    const teknisiName = row.dataset.teknisiName || '';
                    const teknisiEstimatedAt = row.dataset.teknisiEstimatedAt || '';
                    
                    // Format currency
                    const formatCurrency = (value) => {
                        return new Intl.NumberFormat('id-ID', {
                            style: 'currency',
                            currency: 'IDR',
                            minimumFractionDigits: 0
                        }).format(value);
                    };
                    
                    // Fungsi untuk membuat tabel breakdown
                    const createBreakdownTable = (items, type) => {
                        if (!items || items.length === 0) {
                            return '<p class="text-sm text-gray-500 dark:text-gray-400 italic bg-gray-50 dark:bg-gray-800/50 p-3 rounded-lg text-center">Tidak ada data</p>';
                        }
                        return `
                            <div class="overflow-x-auto rounded-lg border border-gray-200 dark:border-gray-600">
                                <table class="w-full border-collapse text-sm">
                                    <thead>
                                        <tr class="bg-gradient-to-r from-gray-100 to-gray-200 dark:from-gray-700 dark:to-gray-800">
                                            <th class="px-4 py-3 text-left text-xs font-bold text-gray-700 dark:text-gray-300 uppercase tracking-wider border-b-2 border-gray-300 dark:border-gray-600">Nama</th>
                                            <th class="px-4 py-3 text-right text-xs font-bold text-gray-700 dark:text-gray-300 uppercase tracking-wider border-b-2 border-gray-300 dark:border-gray-600">Qty</th>
                                            <th class="px-4 py-3 text-right text-xs font-bold text-gray-700 dark:text-gray-300 uppercase tracking-wider border-b-2 border-gray-300 dark:border-gray-600">Harga/Unit</th>
                                            <th class="px-4 py-3 text-right text-xs font-bold text-gray-700 dark:text-gray-300 uppercase tracking-wider border-b-2 border-gray-300 dark:border-gray-600">Total</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        ${items.map((item, index) => `
                                            <tr class="border-b border-gray-200 dark:border-gray-700 ${index % 2 === 0 ? 'bg-white dark:bg-gray-800' : 'bg-gray-50 dark:bg-gray-800/50'} hover:bg-blue-50 dark:hover:bg-blue-900/20 transition-colors">
                                                <td class="px-4 py-3 text-gray-900 dark:text-gray-100 font-medium">${item.name || item.item_name || '-'}</td>
                                                <td class="px-4 py-3 text-right text-gray-700 dark:text-gray-300">${(item.qty || item.quantity || 0).toLocaleString('id-ID')} ${item.unit || ''}</td>
                                                <td class="px-4 py-3 text-right text-gray-700 dark:text-gray-300">${formatCurrency(item.unit_cost || item.cost || 0)}</td>
                                                <td class="px-4 py-3 text-right text-gray-900 dark:text-gray-100 font-bold">${formatCurrency(item.total_cost || ((item.qty || item.quantity || 0) * (item.unit_cost || item.cost || 0)))}</td>
                                            </tr>
                                        `).join('')}
                                    </tbody>
                                </table>
                            </div>
                        `;
                    };
                    
                    // Tampilkan estimasi teknisi jika status admin_review atau sudah ada estimasi
                    const hasEstimation = status === 'admin_review' || teknisiEstimatedAt || materialsBreakdown.length > 0 || sparepartsBreakdown.length > 0 || laborCost > 0;

                    document.getElementById('inquiryDetails').innerHTML = `
                <div class="space-y-5">
                    <!-- Customer Information Card -->
                    <div class="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-5 shadow-sm">
                        <div class="flex items-center mb-4">
                            <div class="p-2.5 bg-blue-100 dark:bg-blue-900/40 rounded-lg mr-3">
                                <i class="fas fa-user text-blue-600 dark:text-blue-400 text-sm"></i>
                            </div>
                            <h3 class="text-base font-bold text-gray-800 dark:text-white">Informasi Customer</h3>
                        </div>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div class="bg-white dark:bg-gray-800/50 rounded-lg p-3 border border-gray-200 dark:border-gray-700">
                                <label class="block text-xs font-semibold text-gray-500 dark:text-gray-400 mb-1.5 uppercase tracking-wide">Nama</label>
                                <p class="text-sm text-gray-900 dark:text-gray-100 font-semibold">${name}</p>
                        </div>
                        <div class="bg-white dark:bg-gray-800/50 rounded-lg p-3 border border-gray-200 dark:border-gray-700">
                                <label class="block text-xs font-semibold text-gray-500 dark:text-gray-400 mb-1.5 uppercase tracking-wide">Email</label>
                                <p class="text-sm text-gray-900 dark:text-gray-100 break-all font-medium">${email}</p>
                        </div>
                        <div class="bg-white dark:bg-gray-800/50 rounded-lg p-3 border border-gray-200 dark:border-gray-700">
                                <label class="block text-xs font-semibold text-gray-500 dark:text-gray-400 mb-1.5 uppercase tracking-wide">Telepon</label>
                                <p class="text-sm text-gray-900 dark:text-gray-100 font-medium">${phone}</p>
                        </div>
                        <div class="bg-white dark:bg-gray-800/50 rounded-lg p-3 border border-gray-200 dark:border-gray-700">
                                <label class="block text-xs font-semibold text-gray-500 dark:text-gray-400 mb-1.5 uppercase tracking-wide">Tanggal</label>
                                <p class="text-sm text-gray-900 dark:text-gray-100 font-medium">${date}</p>
                        </div>
                    </div>
                    </div>

                    <!-- Tracking & Product Card -->
                    <div class="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-5 shadow-sm">
                        <div class="flex items-center mb-4">
                            <div class="p-2.5 bg-indigo-100 dark:bg-indigo-900/40 rounded-lg mr-3">
                                <i class="fas fa-box text-indigo-600 dark:text-indigo-400 text-sm"></i>
                            </div>
                            <h3 class="text-base font-bold text-gray-800 dark:text-white">Informasi Produk</h3>
                        </div>
                        <div class="space-y-3">
                    <div class="bg-gray-50 dark:bg-gray-700/50 rounded-lg p-3 border border-gray-200 dark:border-gray-600">
                                <label class="block text-xs font-semibold text-gray-500 dark:text-gray-400 mb-1.5 uppercase tracking-wide">Tracking Code</label>
                                <span class="inline-flex items-center px-3 py-1.5 rounded-lg text-sm font-semibold bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 border border-gray-300 dark:border-gray-600">
                                    <i class="fas fa-hashtag mr-1.5 text-xs"></i>${tracking}
                                </span>
                    </div>
                    <div class="bg-gray-50 dark:bg-gray-700/50 rounded-lg p-3 border border-gray-200 dark:border-gray-600">
                                <label class="block text-xs font-semibold text-gray-500 dark:text-gray-400 mb-1.5 uppercase tracking-wide">Produk</label>
                                <p class="text-sm text-gray-900 dark:text-gray-100 font-bold">${product}</p>
                    </div>
                    <div class="bg-gray-50 dark:bg-gray-700/50 rounded-lg p-3 border border-gray-200 dark:border-gray-600">
                                <label class="block text-xs font-semibold text-gray-500 dark:text-gray-400 mb-1.5 uppercase tracking-wide">Jenis Permintaan</label>
                                <span class="inline-flex items-center px-3 py-1.5 text-xs font-bold rounded-lg ${badgeCls}">
                                    ${inquiryType === 'catalog' ? '<i class="fas fa-box mr-1.5"></i>' : (inquiryType === 'custom_addon' ? '<i class="fas fa-tools mr-1.5"></i>' : '<i class="fas fa-plus-circle mr-1.5"></i>')}
                                    ${badgeText}
                                </span>
                    </div>
                        </div>
                    </div>
                    
                    ${workflowRecommendation && status === 'pending' && !hasEstimation ? `
                                    <!-- Rekomendasi Alur Card -->
                                    <div class="bg-blue-50 dark:bg-blue-900/20 rounded-lg border-l-4 border-blue-500 dark:border-blue-600 p-5">
                                        <div class="flex items-start">
                                            <div class="flex-shrink-0 w-10 h-10 bg-blue-100 dark:bg-blue-900/40 rounded-lg flex items-center justify-center mr-4">
                                                <i class="fas fa-lightbulb text-blue-600 dark:text-blue-400 text-lg"></i>
                                            </div>
                                            <div class="flex-1">
                                                <h4 class="font-bold text-gray-800 dark:text-gray-200 mb-2 text-base">
                                                    Rekomendasi Alur Penawaran
                                                </h4>
                                                <p class="text-sm text-gray-600 dark:text-gray-400 mb-4 leading-relaxed">
                                                    ${workflowRecommendation.message}
                                                </p>
                                                <div class="flex gap-3">
                                                    ${workflowRecommendation.skip_teknisi ? `
                                                        <button type="button" onclick="forwardToSupervisorFromModal(${id})"
                                                            class="inline-flex items-center px-4 py-2.5 bg-blue-600 hover:bg-blue-700 text-white text-sm font-semibold rounded-lg transition-all duration-200 shadow-sm hover:shadow">
                                                            <i class="fas fa-arrow-right mr-2"></i>
                                                            Forward ke Supervisor
                                                        </button>
                                                    ` : `
                                                        <button type="button" onclick="forwardToTeknisiFromModal(${id})"
                                                            class="inline-flex items-center px-4 py-2.5 bg-green-600 hover:bg-green-700 text-white text-sm font-semibold rounded-lg transition-all duration-200 shadow-sm hover:shadow">
                                                            <i class="fas fa-arrow-right mr-2"></i>
                                                            Forward ke Teknisi
                                                        </button>
                                                    `}
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    ` : ''}

                    ${parsed.addons && parsed.addons.length ? `
                                    <!-- Spesifikasi Tambahan Card -->
                                    <div class="bg-white dark:bg-gray-800 rounded-lg border-l-4 border-yellow-500 dark:border-yellow-600 p-5 shadow-sm">
                                        <div class="flex items-center mb-4">
                                            <div class="p-2.5 bg-yellow-100 dark:bg-yellow-900/40 rounded-lg mr-3">
                                                <i class="fas fa-tools text-yellow-600 dark:text-yellow-400 text-sm"></i>
                                            </div>
                                            <h3 class="text-base font-bold text-gray-800 dark:text-white">Spesifikasi Tambahan (Custom Add-ons)</h3>
                                        </div>
                                        <div class="bg-gray-50 dark:bg-gray-700/50 rounded-lg border border-gray-200 dark:border-gray-600 overflow-hidden">
                                            <div class="overflow-x-auto">
                                                <table class="w-full border-collapse">
                                                    <thead>
                                                        <tr class="bg-gray-100 dark:bg-gray-700">
                                                            <th class="px-4 py-3 text-left text-xs font-bold text-gray-700 dark:text-gray-300 uppercase border-b border-gray-300 dark:border-gray-600">Nama</th>
                                                            <th class="px-4 py-3 text-left text-xs font-bold text-gray-700 dark:text-gray-300 uppercase border-b border-gray-300 dark:border-gray-600">Nilai</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        ${parsed.addons.map(a => `
                                                            <tr class="border-b border-gray-200 dark:border-gray-600 hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors">
                                                                <td class="px-4 py-3 text-sm font-semibold text-gray-900 dark:text-gray-200">${(a?.name||'-')}</td>
                                                                <td class="px-4 py-3 text-sm text-gray-700 dark:text-gray-300 font-medium">${(a?.value||'-')}</td>
                                                            </tr>
                                                        `).join('')}
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>
                                    </div>
                                    ` : ''}

                    <!-- Message Card -->
                    <div class="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-5 shadow-sm">
                        <div class="flex items-center mb-4">
                            <div class="p-2.5 bg-gray-100 dark:bg-gray-700 rounded-lg mr-3">
                                <i class="fas fa-comment-dots text-gray-600 dark:text-gray-400 text-sm"></i>
                    </div>
                            <h3 class="text-base font-bold text-gray-800 dark:text-white">Pesan</h3>
                        </div>
                        <div class="bg-gray-50 dark:bg-gray-700/50 rounded-lg p-4 border border-gray-200 dark:border-gray-600">
                            <textarea class="w-full px-4 py-3 border-0 bg-transparent text-gray-900 dark:text-white resize-none text-sm leading-relaxed focus:outline-none" rows="6" readonly>${parsed.plain || message}</textarea>
                        </div>
                    </div>

                    ${hasEstimation ? `
                                    <!-- Estimasi Teknisi Card -->
                                    <div class="bg-white dark:bg-gray-800 rounded-lg border-l-4 border-blue-500 dark:border-blue-600 p-5 shadow-sm">
                                        <div class="flex items-center justify-between mb-4">
                                            <div class="flex items-center">
                                                <div class="p-2.5 bg-blue-100 dark:bg-blue-900/40 rounded-lg mr-3">
                                                    <i class="fas fa-calculator text-blue-600 dark:text-blue-400 text-sm"></i>
                                                </div>
                                                <h3 class="text-base font-bold text-gray-800 dark:text-white">Estimasi Teknisi</h3>
                                            </div>
                                            ${teknisiEstimatedAt ? `
                                                <span class="text-xs font-semibold text-gray-600 dark:text-gray-400 bg-gray-100 dark:bg-gray-700 px-3 py-1.5 rounded-lg">
                                                    <i class="fas fa-clock mr-1.5"></i>${teknisiEstimatedAt}
                                                    ${teknisiName ? ` oleh ${teknisiName}` : ''}
                                                </span>
                                            ` : ''}
                                        </div>
                                        
                                        <div class="space-y-5 bg-gray-50 dark:bg-gray-700/30 rounded-lg p-4 border border-gray-200 dark:border-gray-600">
                                            ${estimatedDays > 0 ? `
                                                <div class="flex items-center justify-between py-3 px-4 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-600">
                                                    <span class="text-sm font-semibold text-gray-700 dark:text-gray-300">
                                                        <i class="fas fa-calendar-alt mr-2 text-blue-600 dark:text-blue-400"></i>Estimasi Durasi
                                                    </span>
                                                    <span class="text-sm text-gray-900 dark:text-gray-100 font-bold bg-gray-100 dark:bg-gray-700 px-3 py-1 rounded-lg">${estimatedDays} hari</span>
                                                </div>
                                            ` : ''}
                                            
                                            ${materialsBreakdown.length > 0 ? `
                                                <div class="bg-white dark:bg-gray-800 rounded-lg p-4 border border-gray-200 dark:border-gray-600">
                                                    <h4 class="text-xs font-bold text-gray-700 dark:text-gray-300 uppercase mb-3 flex items-center">
                                                        <i class="fas fa-box mr-2 text-blue-600 dark:text-blue-400"></i>Breakdown Bahan Material
                                                    </h4>
                                                    ${createBreakdownTable(materialsBreakdown, 'material')}
                                                    <div class="mt-3 pt-3 border-t border-gray-300 dark:border-gray-600 text-right">
                                                        <span class="text-sm font-bold text-gray-900 dark:text-gray-100 bg-gray-100 dark:bg-gray-700 px-3 py-1.5 rounded-lg inline-block">
                                                            Subtotal: ${formatCurrency(materialsTotal)}
                                                        </span>
                                                    </div>
                                                </div>
                                            ` : ''}
                                            
                                            ${sparepartsBreakdown.length > 0 ? `
                                                <div class="bg-white dark:bg-gray-800 rounded-lg p-4 border border-gray-200 dark:border-gray-600">
                                                    <h4 class="text-xs font-bold text-gray-700 dark:text-gray-300 uppercase mb-3 flex items-center">
                                                        <i class="fas fa-cog mr-2 text-blue-600 dark:text-blue-400"></i>Breakdown Sparepart
                                                    </h4>
                                                    ${createBreakdownTable(sparepartsBreakdown, 'sparepart')}
                                                    <div class="mt-3 pt-3 border-t border-gray-300 dark:border-gray-600 text-right">
                                                        <span class="text-sm font-bold text-gray-900 dark:text-gray-100 bg-gray-100 dark:bg-gray-700 px-3 py-1.5 rounded-lg inline-block">
                                                            Subtotal: ${formatCurrency(sparepartsTotal)}
                                                        </span>
                                                    </div>
                                                </div>
                                            ` : ''}
                                            
                                            ${laborCostTotal > 0 ? `
                                                <div class="flex items-center justify-between py-3 px-4 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-600">
                                                    <span class="text-sm font-semibold text-gray-700 dark:text-gray-300">
                                                        <i class="fas fa-users mr-2 text-blue-600 dark:text-blue-400"></i>Biaya Tenaga Kerja
                                                    </span>
                                                    <span class="text-sm text-gray-900 dark:text-gray-100 font-bold bg-gray-100 dark:bg-gray-700 px-3 py-1 rounded-lg">${formatCurrency(laborCostTotal)}</span>
                                                </div>
                                            ` : ''}
                                            
                                            <div class="bg-blue-600 dark:bg-blue-700 rounded-lg p-4 shadow-sm mt-4">
                                                <div class="flex items-center justify-between mb-2">
                                                    <span class="text-sm font-bold text-white">Total Biaya Internal</span>
                                                    <span class="text-lg font-extrabold text-white">${formatCurrency(totalCost || (materialsTotal + sparepartsTotal + laborCostTotal))}</span>
                                                </div>
                                                ${engineerSuggested > 0 ? `
                                                    <div class="flex items-center justify-between pt-3 mt-3 border-t border-blue-400 dark:border-blue-500">
                                                        <span class="text-xs font-semibold text-blue-100">Usulan Harga Teknisi</span>
                                                        <span class="text-sm font-bold text-white bg-blue-500 dark:bg-blue-600 px-3 py-1 rounded-lg">${formatCurrency(engineerSuggested)}</span>
                                                    </div>
                                                ` : ''}
                                            </div>
                                            
                                            ${teknisiNote ? `
                                                <div class="mt-4 p-4 bg-yellow-50 dark:bg-yellow-900/20 rounded-lg border-l-4 border-yellow-400 dark:border-yellow-600">
                                                    <h4 class="text-xs font-bold text-gray-700 dark:text-gray-300 uppercase mb-2 flex items-center">
                                                        <i class="fas fa-sticky-note mr-2 text-yellow-600 dark:text-yellow-400"></i>Catatan Teknisi
                                                    </h4>
                                                    <p class="text-sm text-gray-700 dark:text-gray-300 whitespace-pre-wrap leading-relaxed">${teknisiNote}</p>
                                                </div>
                                            ` : ''}
                                        </div>
                                    </div>
                                    ` : ''}
                                    
                    ${status === 'admin_review' && hasEstimation ? `
                                    <!-- Admin Actions Card -->
                                    <div class="bg-yellow-50 dark:bg-yellow-900/20 rounded-lg border-l-4 border-yellow-500 dark:border-yellow-600 p-5">
                                        <div class="flex items-start">
                                            <div class="flex-shrink-0 w-10 h-10 bg-yellow-100 dark:bg-yellow-900/40 rounded-lg flex items-center justify-center mr-4">
                                                <i class="fas fa-tasks text-yellow-600 dark:text-yellow-400 text-lg"></i>
                                            </div>
                                            <div class="flex-1">
                                                <h4 class="font-bold text-gray-800 dark:text-gray-200 mb-2 text-base">
                                                    Tindakan Admin
                                                </h4>
                                                <p class="text-sm text-gray-600 dark:text-gray-400 mb-4 leading-relaxed">
                                                    Review estimasi teknisi. Jika perlu revisi, klik tombol di bawah.
                                                </p>
                                                <div class="flex gap-3 flex-wrap">
                                                    <button type="button" onclick="openRevisionModal(${id})"
                                                        class="inline-flex items-center px-4 py-2.5 bg-yellow-600 hover:bg-yellow-700 text-white text-sm font-semibold rounded-lg transition-all duration-200 shadow-sm hover:shadow">
                                                        <i class="fas fa-edit mr-2"></i>
                                                        Minta Revisi
                                                    </button>
                                                    <button type="button" onclick="forwardToSupervisorFromModal(${id})"
                                                        class="inline-flex items-center px-4 py-2.5 bg-green-600 hover:bg-green-700 text-white text-sm font-semibold rounded-lg transition-all duration-200 shadow-sm hover:shadow">
                                                        <i class="fas fa-check-circle mr-2"></i>
                                                        Setujui & Forward ke Supervisor
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    ` : ''}
                                    
                    ${row.dataset.adminRevisionNote ? `
                                    <!-- Admin Revision Note Card -->
                                    <div class="bg-red-50 dark:bg-red-900/20 rounded-lg border-l-4 border-red-500 dark:border-red-600 p-5">
                                        <div class="flex items-start">
                                            <div class="flex-shrink-0 p-2.5 bg-red-100 dark:bg-red-900/40 rounded-lg mr-4">
                                                <i class="fas fa-comment-alt text-red-600 dark:text-red-400 text-sm"></i>
                                            </div>
                                            <div class="flex-1">
                                                <div class="flex items-center justify-between mb-3">
                                                    <h4 class="font-bold text-gray-800 dark:text-gray-200 text-base">
                                                        Catatan Revisi dari Admin
                                                    </h4>
                                                    ${row.dataset.adminRevisionAt ? `
                                                        <span class="text-xs font-semibold text-gray-600 dark:text-gray-400 bg-gray-100 dark:bg-gray-700 px-3 py-1.5 rounded-lg">
                                                            <i class="fas fa-clock mr-1.5"></i>${row.dataset.adminRevisionAt}
                                                        </span>
                                                    ` : ''}
                                                </div>
                                                <div class="bg-white dark:bg-gray-800 rounded-lg p-4 border border-gray-200 dark:border-gray-600">
                                                    <p class="text-sm text-gray-700 dark:text-gray-300 whitespace-pre-wrap leading-relaxed">${row.dataset.adminRevisionNote}</p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    ` : ''}

                    ${hasReply ? `
                                    <!-- Admin Reply Card -->
                                    <div class="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-5 shadow-sm">
                                        <div class="flex items-center mb-3">
                                            <div class="p-2.5 bg-green-100 dark:bg-green-900/40 rounded-lg mr-3">
                                                <i class="fas fa-reply text-green-600 dark:text-green-400 text-sm"></i>
                                            </div>
                                            <h3 class="text-base font-bold text-gray-800 dark:text-white">Balasan Admin</h3>
                                        </div>
                                        <div class="bg-gray-50 dark:bg-gray-700/50 rounded-lg p-3 border border-gray-200 dark:border-gray-600">
                                            <p class="text-sm text-gray-700 dark:text-gray-300 font-medium flex items-center">
                                                <i class="fas fa-check-circle text-green-500 mr-2"></i>Penawaran sudah direspon oleh admin
                                            </p>
                                        </div>
                                    </div>
                                    ` : ''}
                </div>
            `;

                    // Sembunyikan tombol mark as read jika sudah dibaca
                    const markBtn = document.getElementById('markAsReadBtn');
                    if (markBtn) {
                        if (isRead === 1) {
                            markBtn.style.display = 'none';
                        } else {
                            markBtn.style.display = 'block';
                        }
                    }

                    const modal = document.getElementById('inquiryModal');
                    const modalContent = document.getElementById('inquiryModalContent');

                    if (modal && modalContent) {
                        // Reset transform
                        modalContent.style.transform = 'translateX(-100%)';
                        modalContent.style.opacity = '0';

                        // Show modal
                        modal.classList.remove('hidden');
                        modal.classList.add('flex', 'items-center', 'justify-center');
                        document.body.style.overflow = 'hidden';

                        // Slide in animation
                        setTimeout(() => {
                            modalContent.style.transform = 'translateX(0)';
                            modalContent.style.opacity = '1';
                        }, 10);
                    }
                }
            }

            function closeInquiryModal() {
                const modal = document.getElementById('inquiryModal');
                const modalContent = document.getElementById('inquiryModalContent');

                if (modal && modalContent) {
                    // Slide out animation
                    modalContent.style.transform = 'translateX(-100%)';
                    modalContent.style.opacity = '0';

                    // Hide modal after animation
                    setTimeout(() => {
                        modal.classList.remove('flex', 'items-center', 'justify-center');
                        modal.classList.add('hidden');
                        modalContent.style.transform = '';
                        modalContent.style.opacity = '';
                        document.body.style.overflow = 'auto';
                    }, 300);
                }
            }

            function openInquiryModalAndReply() {
                closeInquiryModal();
                setTimeout(() => {
                    openReplyModal(currentInquiryId);
                }, 300);
            }

            function openReplyModalFromDetail() {
                closeInquiryModal();
                setTimeout(() => {
                    openReplyModal(currentInquiryId);
                }, 300);
            }

            function markAsRead(inquiryId) {
                const token = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
                if (!token) {
                    Swal.fire('Error!', 'CSRF token tidak ditemukan. Refresh halaman.', 'error');
                    return;
                }

                Swal.fire({
                    title: 'Tandai sebagai Dibaca?',
                    text: 'Apakah Anda yakin ingin menandai penawaran ini sebagai sudah dibaca?',
                    icon: 'question',
                    showCancelButton: true,
                    confirmButtonText: '<i class="fas fa-check mr-2"></i>Ya, Tandai Dibaca',
                    cancelButtonText: '<i class="fas fa-times mr-2"></i>Batal',
                    confirmButtonColor: '#10b981',
                    cancelButtonColor: '#6b7280',
                    reverseButtons: true,
                    customClass: {
                        popup: 'rounded-lg',
                        confirmButton: 'px-4 py-2 rounded-md font-medium',
                        cancelButton: 'px-4 py-2 rounded-md font-medium'
                    }
                }).then((result) => {
                    if (result.isConfirmed) {
                        fetch(`/admin/product-inquiries/${inquiryId}/read`, {
                                method: 'POST',
                                headers: {
                                    'X-CSRF-TOKEN': token,
                                    'Content-Type': 'application/json',
                                    'X-Requested-With': 'XMLHttpRequest'
                                }
                            })
                            .then(response => response.json())
                            .then(data => {
                                if (data.success) {
                                    // Refresh notification system untuk update badge
                                    if (window.notificationSystem && typeof window.notificationSystem
                                        .checkNotifications === 'function') {
                                        setTimeout(() => {
                                            window.notificationSystem.checkNotifications();
                                        }, 300);
                                    }

                                    Swal.fire({
                                        title: 'Berhasil!',
                                        text: data.message,
                                        icon: 'success',
                                        timer: 2000,
                                        showConfirmButton: false
                                    });
                                    setTimeout(() => {
                                        window.location.reload();
                                    }, 2000);
                                } else {
                                    Swal.fire('Error!', data.message || 'Gagal menandai sebagai dibaca', 'error');
                                }
                            })
                            .catch(error => {
                                console.error('Error:', error);
                                Swal.fire('Error!', 'Terjadi kesalahan saat memproses', 'error');
                            });
                    }
                });
            }

            function markAsReadFromModal() {
                if (!currentInquiryId) return;

                const token = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
                if (!token) {
                    Swal.fire('Error!', 'CSRF token tidak ditemukan. Refresh halaman.', 'error');
                    return;
                }

                const submitBtn = document.getElementById('markAsReadBtn');
                submitBtn.disabled = true;
                submitBtn.textContent = 'Memproses...';

                fetch(`/admin/product-inquiries/${currentInquiryId}/read`, {
                        method: 'POST',
                        headers: {
                            'X-CSRF-TOKEN': token,
                            'Content-Type': 'application/json',
                            'X-Requested-With': 'XMLHttpRequest'
                        }
                    })
                    .then(async (response) => {
                        if (response.status === 419) {
                            throw new Error('CSRF token mismatch. Refresh halaman atau login ulang.');
                        }
                        if (!response.ok) {
                            const errorText = await response.text();
                            throw new Error(`HTTP ${response.status}: ${errorText.substring(0, 100)}`);
                        }
                        return response.json();
                    })
                    .then((data) => {
                        if (data.success) {
                            // Refresh notification system untuk update badge
                            if (window.notificationSystem && typeof window.notificationSystem.checkNotifications ===
                                'function') {
                                setTimeout(() => {
                                    window.notificationSystem.checkNotifications();
                                }, 300);
                            }

                            Swal.fire({
                                title: 'Berhasil!',
                                text: data.message || 'Penawaran ditandai sebagai dibaca.',
                                icon: 'success',
                                timer: 1500,
                                showConfirmButton: false
                            }).then(() => {
                                location.reload();
                            });
                        } else {
                            Swal.fire('Error!', data.message || 'Gagal update status.', 'error');
                        }
                    })
                    .catch((error) => {
                        console.error('Full error:', error);
                        Swal.fire('Error!', 'Gagal kirim: ' + error.message + ' (Cek console untuk detail)', 'error');
                    })
                    .finally(() => {
                        submitBtn.disabled = false;
                        submitBtn.textContent = 'Tandai Dibaca';
                    });
            }

            // Modal Reply Functions
            function openReplyModal(inquiryId) {
                currentInquiryId = inquiryId;
                const row = document.querySelector(`tr[data-inquiry-id="${inquiryId}"]`);
                if (row) {
                    const name = row.dataset.inquiryName;
                    const email = row.dataset.inquiryEmail;

                    document.getElementById('reply-inquiry-id').value = inquiryId;
                    document.getElementById('replyTo').textContent = `${name} (${email})`;

                    // Reset form values dengan default subject
                    document.getElementById('replySubject').value = 'Penawaran Produk';
                    document.getElementById('replyMessage').value = '';
                    document.getElementById('replyMethod').value = '';
                }

                const modal = document.getElementById('replyModal');
                const modalContent = document.getElementById('replyModalContent');

                if (modal && modalContent) {
                    // Reset transform
                    modalContent.style.transform = 'translateX(100%)';
                    modalContent.style.opacity = '0';

                    // Show modal
                    modal.classList.remove('hidden');
                    modal.classList.add('flex', 'items-center', 'justify-center');
                    document.body.style.overflow = 'hidden';

                    // Slide in animation
                    setTimeout(() => {
                        modalContent.style.transform = 'translateX(0)';
                        modalContent.style.opacity = '1';
                    }, 10);
                }

                // Setup event listeners untuk tombol setelah modal dibuka
                setupReplyModalListeners();
            }

            function setupReplyModalListeners() {
                // Remove all existing event listeners dengan membuat elemen baru
                const kirimBtn = document.getElementById('kirimReplyBtn');
                const batalBtn = document.getElementById('batalReplyBtn');

                if (kirimBtn) {
                    // Clone dan replace untuk remove semua event listeners
                    const newKirimBtn = kirimBtn.cloneNode(true);
                    kirimBtn.parentNode.replaceChild(newKirimBtn, kirimBtn);

                    // Add event listener ke elemen baru
                    document.getElementById('kirimReplyBtn').addEventListener('click', function(e) {
                        e.preventDefault();
                        e.stopPropagation();
                        e.stopImmediatePropagation();
                        sendReply(e);
                        return false;
                    }, true); // Use capture phase untuk memastikan event ter-handle
                }

                if (batalBtn) {
                    // Clone dan replace untuk remove semua event listeners
                    const newBatalBtn = batalBtn.cloneNode(true);
                    batalBtn.parentNode.replaceChild(newBatalBtn, batalBtn);

                    // Add event listener ke elemen baru
                    document.getElementById('batalReplyBtn').addEventListener('click', function(e) {
                        e.preventDefault();
                        e.stopPropagation();
                        e.stopImmediatePropagation();
                        closeReplyModal(e);
                        return false;
                    }, true); // Use capture phase
                }

                // Setup event listener untuk tombol Selesai
                const selesaiBtn = document.getElementById('selesaiReplyBtn');
                if (selesaiBtn) {
                    // Clone untuk remove existing listeners
                    const newSelesaiBtn = selesaiBtn.cloneNode(true);
                    selesaiBtn.parentNode.replaceChild(newSelesaiBtn, selesaiBtn);

                    // Add fresh event listener
                    document.getElementById('selesaiReplyBtn').addEventListener('click', function(e) {
                        e.preventDefault();
                        e.stopPropagation();
                        e.stopImmediatePropagation();

                        // Close modal dengan animasi
                        closeReplyModal(e);

                        // Reload setelah modal tertutup
                        setTimeout(() => {
                            window.location.reload();
                        }, 350);

                        return false;
                    }, true);
                }
            }

            function closeReplyModal(event) {
                // Prevent default behavior jika ada event
                if (event) {
                    event.preventDefault();
                    event.stopPropagation();
                    event.stopImmediatePropagation();
                }

                // Reset form dan loading state saat tutup modal
                const defaultButtons = document.getElementById('defaultButtons');
                const selesaiBtn = document.getElementById('selesaiReplyBtn');
                const kirimBtn = document.getElementById('kirimReplyBtn');
                const kirimIcon = document.getElementById('kirimReplyIcon');
                const kirimText = document.getElementById('kirimReplyText');
                const kirimLoading = document.getElementById('kirimReplyLoading');

                // Reset tombol ke state default
                if (defaultButtons) defaultButtons.classList.remove('hidden');
                if (selesaiBtn) selesaiBtn.classList.add('hidden');
                if (kirimBtn) kirimBtn.disabled = false;
                if (kirimIcon) {
                    kirimIcon.classList.remove('hidden');
                    kirimIcon.classList.remove('fa-check');
                    kirimIcon.classList.add('fa-paper-plane');
                }
                if (kirimText) kirimText.classList.remove('hidden');
                if (kirimLoading) kirimLoading.classList.add('hidden');

                // Reset form
                const replyForm = document.getElementById('replyForm');
                if (replyForm) {
                    const form = replyForm.closest('form') || replyForm;
                    if (form && form.reset) form.reset();
                }

                const modal = document.getElementById('replyModal');
                const modalContent = document.getElementById('replyModalContent');

                if (modal && modalContent) {
                    // Slide out animation
                    modalContent.style.transform = 'translateX(100%)';
                    modalContent.style.opacity = '0';

                    // Hide modal after animation
                    setTimeout(() => {
                        modal.classList.remove('flex', 'items-center', 'justify-center');
                        modal.classList.add('hidden');
                        modalContent.style.transform = '';
                        modalContent.style.opacity = '';
                        document.body.style.overflow = 'auto';
                    }, 300);
                }

                return false;
            }

            function sendReply(event) {
                // Prevent default behavior jika ada event
                if (event) {
                    event.preventDefault();
                    event.stopPropagation();
                    event.stopImmediatePropagation();
                }

                const subject = document.getElementById('replySubject').value.trim();
                const message = document.getElementById('replyMessage').value.trim();
                const replyMethod = document.getElementById('replyMethod').value;
                const inquiryId = currentInquiryId || document.getElementById('reply-inquiry-id').value;

                if (!subject || !message || !replyMethod) {
                    Swal.fire({
                        title: 'Error!',
                        text: 'Semua field harus diisi!',
                        icon: 'error',
                        confirmButtonText: 'OK',
                        allowOutsideClick: false
                    });
                    return false;
                }

                if (!inquiryId) {
                    Swal.fire({
                        title: 'Error!',
                        text: 'Inquiry ID tidak ditemukan. Silakan tutup dan buka modal lagi.',
                        icon: 'error',
                        confirmButtonText: 'OK',
                        allowOutsideClick: false
                    });
                    return false;
                }

                // Langsung kirim tanpa konfirmasi - hanya tampilkan loading dan notif sukses
                // Set loading state dan sembunyikan tombol default, tampilkan tombol Selesai
                const defaultButtons = document.getElementById('defaultButtons');
                const selesaiBtn = document.getElementById('selesaiReplyBtn');
                const kirimBtn = document.getElementById('kirimReplyBtn');
                const kirimIcon = document.getElementById('kirimReplyIcon');
                const kirimText = document.getElementById('kirimReplyText');
                const kirimLoading = document.getElementById('kirimReplyLoading');

                // Sembunyikan tombol Batal dan Kirim Reply, tampilkan tombol Selesai dengan loading
                if (defaultButtons) defaultButtons.classList.add('hidden');
                if (selesaiBtn) {
                    selesaiBtn.classList.remove('hidden');
                    selesaiBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Mengirim...';
                    selesaiBtn.disabled = true;
                }

                fetch(`/admin/product-inquiries/${inquiryId}/reply`, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                        },
                        body: JSON.stringify({
                            subject: subject,
                            message: message,
                            reply_method: replyMethod
                        })
                    })
                    .then(response => response.json())
                    .then(data => {
                        // Update tombol Selesai - hilangkan loading, tampilkan icon check dan enable tombol
                        const selesaiBtn = document.getElementById('selesaiReplyBtn');
                        if (selesaiBtn) {
                            selesaiBtn.innerHTML = '<i class="fas fa-check mr-2"></i>Selesai';
                            selesaiBtn.disabled = false;
                        }

                        if (data.success) {
                            // Tambahkan delay kecil sebelum menampilkan notifikasi untuk memastikan tidak langsung hilang
                            setTimeout(function() {
                                if (data.whatsapp_link) {
                                    Swal.fire({
                                        title: 'Berhasil!',
                                        text: data.message,
                                        icon: 'success',
                                        showCancelButton: true,
                                        confirmButtonText: 'Buka WhatsApp',
                                        cancelButtonText: 'Tutup',
                                        timer: 0, // Tidak auto close
                                        timerProgressBar: false,
                                        allowOutsideClick: false,
                                        allowEscapeKey: true,
                                        didOpen: () => {
                                            // Pastikan notifikasi tetap terbuka
                                            Swal.stopTimer();
                                        }
                                    }).then((result) => {
                                        if (result.isConfirmed) {
                                            window.open(data.whatsapp_link, '_blank');
                                        }
                                        // Tombol Selesai sudah ada, user bisa klik untuk tutup
                                    });
                                } else if (data.phone_link) {
                                    Swal.fire({
                                        title: 'Berhasil!',
                                        text: data.message,
                                        icon: 'success',
                                        showCancelButton: true,
                                        confirmButtonText: 'Telepon',
                                        cancelButtonText: 'Tutup',
                                        timer: 0, // Tidak auto close
                                        timerProgressBar: false,
                                        allowOutsideClick: false,
                                        allowEscapeKey: true,
                                        didOpen: () => {
                                            // Pastikan notifikasi tetap terbuka
                                            Swal.stopTimer();
                                        }
                                    }).then((result) => {
                                        if (result.isConfirmed) {
                                            window.location.href = data.phone_link;
                                        }
                                        // Tombol Selesai sudah ada, user bisa klik untuk tutup
                                    });
                                } else {
                                    // Tampilkan notifikasi sukses - tidak akan auto close
                                    // Tutup semua Swal yang mungkin masih terbuka terlebih dahulu
                                    if (typeof Swal !== 'undefined' && Swal.isVisible()) {
                                        Swal.close();
                                    }

                                    // Tunggu sebentar untuk memastikan tidak ada konflik
                                    setTimeout(function() {
                                        // Pastikan tidak ada Swal lain yang terbuka
                                        const existingSwal = document.querySelector('.swal2-container');
                                        if (existingSwal && !existingSwal.classList.contains(
                                                'swal2-backdrop-show')) {
                                            existingSwal.remove();
                                        }

                                        // Simpan referensi Swal untuk mencegah auto close
                                        let swalInstance = null;

                                        // Tampilkan notifikasi sukses - Auto close setelah 3 detik, atau bisa ditutup dengan klik OK
                                        swalInstance = Swal.fire({
                                            title: 'Berhasil!',
                                            text: data.message,
                                            icon: 'success',
                                            confirmButtonText: 'OK',
                                            showConfirmButton: true,
                                            timer: 6000, // Auto close setelah 6 detik (6000ms)
                                            timerProgressBar: true, // Tampilkan progress bar countdown
                                            allowOutsideClick: false, // Tidak bisa tutup dengan klik di luar
                                            allowEscapeKey: false, // Tidak bisa tutup dengan ESC
                                            allowEnterKey: true, // Bisa tutup dengan Enter saat focus di tombol OK
                                            focusConfirm: true,
                                            backdrop: true,
                                            showCloseButton: false, // Tidak ada tombol X
                                            didOpen: (popup) => {
                                                // Timer sudah di-set ke 3000ms, biarkan berjalan
                                                // User bisa klik OK kapan saja untuk menutup lebih cepat

                                                // Prevent click pada backdrop
                                                const backdrop = document.querySelector(
                                                    '.swal2-backdrop');
                                                if (backdrop) {
                                                    backdrop.style.pointerEvents = 'none';
                                                    backdrop.onclick = function(e) {
                                                        e.stopPropagation();
                                                        e.preventDefault();
                                                        return false;
                                                    };
                                                }

                                                // Prevent ESC key
                                                const escHandler = function(e) {
                                                    if (e.key === 'Escape' || e.keyCode ===
                                                        27) {
                                                        e.stopPropagation();
                                                        e.preventDefault();
                                                        return false;
                                                    }
                                                };
                                                document.addEventListener('keydown', escHandler,
                                                    true);
                                                popup._escHandler = escHandler;
                                            },
                                            didClose: () => {
                                                // Cleanup event listeners
                                                const popup = document.querySelector(
                                                    '.swal2-popup');
                                                if (popup && popup._escHandler) {
                                                    document.removeEventListener('keydown',
                                                        popup._escHandler, true);
                                                }

                                                // Hanya reload setelah user benar-benar klik OK
                                                const modal = document.getElementById(
                                                    'replyModal');
                                                if (modal) {
                                                    modal.style.display = 'none';
                                                    modal.style.opacity = '0';
                                                    modal.style.visibility = 'hidden';
                                                    modal.classList.add('hidden');
                                                    modal.classList.remove('flex',
                                                        'items-center', 'justify-center');
                                                }
                                                document.body.style.overflow = 'auto';

                                                // Delay kecil sebelum reload
                                                setTimeout(function() {
                                                    window.location.reload();
                                                }, 100);
                                            }
                                        });
                                    }, 200);
                                    // Tombol Selesai sudah tampil, user bisa klik untuk tutup dan reload
                                }
                            }, 300); // Delay 300ms sebelum menampilkan notifikasi
                        } else {
                            // Jika error, kembalikan ke tombol default
                            const defaultButtons = document.getElementById('defaultButtons');
                            const selesaiBtn = document.getElementById('selesaiReplyBtn');
                            if (defaultButtons) defaultButtons.classList.remove('hidden');
                            if (selesaiBtn) selesaiBtn.classList.add('hidden');

                            Swal.fire({
                                title: 'Error!',
                                text: data.message,
                                icon: 'error',
                                confirmButtonText: 'OK'
                            });
                        }
                    })
                    .catch(error => {
                        console.error('Error:', error);
                        // Jika error, kembalikan ke tombol default
                        const defaultButtons = document.getElementById('defaultButtons');
                        const selesaiBtn = document.getElementById('selesaiReplyBtn');
                        if (defaultButtons) defaultButtons.classList.remove('hidden');
                        if (selesaiBtn) selesaiBtn.classList.add('hidden');

                        Swal.fire({
                            title: 'Error!',
                            text: 'Terjadi kesalahan saat mengirim reply',
                            icon: 'error',
                            confirmButtonText: 'OK'
                        });
                    });
            }

            // Tutup modal jika klik di luar (tapi jangan tutup jika Swal terbuka)
            document.addEventListener('click', function(e) {
                // Jangan tutup modal jika Swal sedang terbuka
                if (typeof Swal !== 'undefined' && Swal.isVisible()) {
                    return;
                }

                const inquiryModal = document.getElementById('inquiryModal');
                const replyModal = document.getElementById('replyModal');
                const finalOfferModal = document.getElementById('finalOfferModal');
                const convertOrderModal = document.getElementById('convertOrderModal');

                if (e.target === inquiryModal) {
                    closeInquiryModal();
                }
                if (e.target === replyModal) {
                    closeReplyModal();
                }
                if (e.target === finalOfferModal) {
                    closeFinalOfferModal();
                }
                if (e.target === convertOrderModal) {
                    closeConvertOrderModal();
                }
            });

            // Keyboard ESC for modal (tapi jangan tutup jika Swal terbuka)
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape') {
                    // Jangan tutup modal jika Swal sedang terbuka
                    if (typeof Swal !== 'undefined' && Swal.isVisible()) {
                        return;
                    }

                    const inquiryModal = document.getElementById('inquiryModal');
                    const replyModal = document.getElementById('replyModal');
                    const finalOfferModal = document.getElementById('finalOfferModal');
                    const convertOrderModal = document.getElementById('convertOrderModal');
                    const revisionModal = document.getElementById('revisionModal');

                    if (inquiryModal && !inquiryModal.classList.contains('hidden')) {
                        closeInquiryModal();
                    }
                    if (replyModal && !replyModal.classList.contains('hidden')) {
                        closeReplyModal();
                    }
                    if (finalOfferModal && !finalOfferModal.classList.contains('hidden')) {
                        closeFinalOfferModal();
                    }
                    if (convertOrderModal && !convertOrderModal.classList.contains('hidden')) {
                        closeConvertOrderModal();
                    }
                    if (revisionModal && !revisionModal.classList.contains('hidden')) {
                        closeRevisionModal();
                    }
                }
            });

            // Submit filter dengan parameter yang bersih
            function submitFilter() {
                const params = new URLSearchParams();

                // Ambil nilai dari setiap input/select
                const search = document.getElementById('searchInput')?.value?.trim() || '';
                const status = document.getElementById('statusFilter')?.value || '';
                const state = document.getElementById('stateFilter')?.value || '';

                // Hanya tambahkan parameter yang memiliki nilai
                if (search) {
                    params.append('search', search);
                }
                if (status) {
                    params.append('status', status);
                }
                if (state) {
                    params.append('state', state);
                }

                // Redirect dengan parameter yang bersih
                const url = '{{ route('admin.product-inquiries.index') }}' + (params.toString() ? '?' + params.toString() :
                    '');
                window.location.href = url;
            }

            // Clear individual filter
            function clearFilter(filterName) {
                const url = new URL(window.location);
                url.searchParams.delete(filterName);
                window.location.href = url.toString();
            }

            // Clear all filters
            function clearAllFilters() {
                window.location.href = '{{ route('admin.product-inquiries.index') }}';
            }

            // Submit form search dengan Enter
            document.addEventListener('DOMContentLoaded', function() {
                const searchInput = document.getElementById('searchInput');
                if (searchInput) {
                    searchInput.addEventListener('keypress', function(e) {
                        if (e.key === 'Enter') {
                            e.preventDefault();
                            submitFilter();
                        }
                    });
                }
            });

            // Confirm Delete Inquiry
            function confirmDeleteInquiry(id) {
                Swal.fire({
                    title: 'Yakin ingin menghapus?',
                    html: '<p class="text-sm text-gray-600 dark:text-gray-400">Penawaran ini akan dihapus secara permanen!</p><p class="text-xs text-red-600 dark:text-red-400 mt-2"><i class="fas fa-exclamation-triangle mr-1"></i>Tindakan ini tidak dapat dibatalkan.</p>',
                    icon: 'warning',
                    showCancelButton: true,
                    confirmButtonColor: '#dc2626',
                    cancelButtonColor: '#6b7280',
                    confirmButtonText: '<i class="fas fa-trash mr-2"></i>Ya, Hapus',
                    cancelButtonText: '<i class="fas fa-times mr-2"></i>Batal',
                    reverseButtons: true,
                    customClass: {
                        popup: 'rounded-lg',
                        confirmButton: 'px-4 py-2 rounded-md font-medium',
                        cancelButton: 'px-4 py-2 rounded-md font-medium'
                    }
                }).then((result) => {
                    if (result.isConfirmed) {
                        document.getElementById('delete-form-' + id).submit();
                    }
                });
            }

            // Bulk Actions Functions
            function toggleSelectAll() {
                const selectAll = document.getElementById('selectAll');
                const checkboxes = document.querySelectorAll('.inquiry-checkbox');

                checkboxes.forEach(checkbox => {
                    checkbox.checked = selectAll.checked;
                });

                updateBulkActionButton();
            }

            function updateBulkActionButton() {
                const checkboxes = document.querySelectorAll('.inquiry-checkbox:checked');
                const bulkActionBtn = document.getElementById('bulkActionBtn');
                const bulkAction = document.getElementById('bulkAction');

                if (checkboxes.length > 0) {
                    bulkActionBtn.disabled = false;
                    bulkActionBtn.classList.remove('disabled:bg-gray-400', 'disabled:cursor-not-allowed');
                    bulkActionBtn.classList.add('bg-blue-600', 'hover:bg-blue-700');
                } else {
                    bulkActionBtn.disabled = true;
                    bulkActionBtn.classList.add('disabled:bg-gray-400', 'disabled:cursor-not-allowed');
                    bulkActionBtn.classList.remove('bg-blue-600', 'hover:bg-blue-700');
                }
            }

            function executeBulkAction() {
                const checkboxes = document.querySelectorAll('.inquiry-checkbox:checked');
                const bulkAction = document.getElementById('bulkAction').value;

                if (checkboxes.length === 0) {
                    Swal.fire({
                        title: 'Error!',
                        text: 'Pilih minimal satu item!',
                        icon: 'error',
                        confirmButtonText: 'OK'
                    });
                    return;
                }

                if (!bulkAction) {
                    Swal.fire({
                        title: 'Error!',
                        text: 'Pilih aksi yang akan dilakukan!',
                        icon: 'error',
                        confirmButtonText: 'OK'
                    });
                    return;
                }

                const inquiryIds = Array.from(checkboxes).map(cb => cb.value);

                let confirmText = '';
                let confirmIcon = 'question';

                switch (bulkAction) {
                    case 'mark-read':
                        confirmText = `Tandai ${inquiryIds.length} penawaran sebagai dibaca?`;
                        confirmIcon = 'question';
                        break;
                    case 'mark-unread':
                        confirmText = `Tandai ${inquiryIds.length} penawaran sebagai belum dibaca?`;
                        confirmIcon = 'question';
                        break;
                    case 'delete':
                        confirmText = `Hapus ${inquiryIds.length} penawaran secara permanen?`;
                        confirmIcon = 'warning';
                        break;
                }

                Swal.fire({
                    title: 'Konfirmasi',
                    text: confirmText,
                    icon: confirmIcon,
                    showCancelButton: true,
                    confirmButtonColor: bulkAction === 'delete' ? '#ef4444' : '#10b981',
                    cancelButtonColor: '#6b7280',
                    confirmButtonText: 'Ya, Lanjutkan',
                    cancelButtonText: 'Batal'
                }).then((result) => {
                    if (result.isConfirmed) {
                        performBulkAction(bulkAction, inquiryIds);
                    }
                });
            }

            function performBulkAction(action, inquiryIds) {
                const url = `/admin/product-inquiries/bulk-${action}`;

                fetch(url, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                        },
                        body: JSON.stringify({
                            inquiry_ids: inquiryIds
                        })
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            // Refresh notification system untuk update badge (khusus untuk mark-read)
                            if (action === 'mark-read' && window.notificationSystem && typeof window.notificationSystem
                                .checkNotifications === 'function') {
                                setTimeout(() => {
                                    window.notificationSystem.checkNotifications();
                                }, 300);
                            }

                            Swal.fire({
                                title: 'Berhasil!',
                                text: data.message,
                                icon: 'success',
                                confirmButtonText: 'OK'
                            });
                            // Reload halaman untuk update data
                            setTimeout(() => {
                                window.location.reload();
                            }, 1000);
                        } else {
                            Swal.fire({
                                title: 'Error!',
                                text: data.message,
                                icon: 'error',
                                confirmButtonText: 'OK'
                            });
                        }
                    })
                    .catch(error => {
                        console.error('Error:', error);
                        Swal.fire({
                            title: 'Error!',
                            text: 'Terjadi kesalahan saat memproses aksi',
                            icon: 'error',
                            confirmButtonText: 'OK'
                        });
                    });
            }
        </script>

        <style>
            /* Konsisten Style untuk semua Index Pages */
            .form-input-search,
            .form-select {
                height: 40px !important;
                min-height: 40px !important;
                max-height: 40px !important;
                box-sizing: border-box !important;
                line-height: 1.5 !important;
                padding: 0 2.5rem 0 2.5rem !important;
                margin: 0 !important;
            }

            .form-input-search {
                width: 100%;
                padding-left: 2.5rem !important;
                padding-right: 2.5rem !important;
                border: 1px solid rgb(209 213 219);
                border-radius: 0.5rem;
                background-color: white;
                color: rgb(17 24 39);
                font-size: 0.875rem;
                transition: all 0.2s;
            }

            .dark .form-input-search {
                border-color: rgb(75 85 99);
                background-color: rgb(55 65 81);
                color: white;
            }

            .form-input-search:focus {
                outline: none;
                ring: 2px;
                ring-color: rgb(59 130 246);
            }

            .form-select {
                width: 100%;
                padding-left: 1rem !important;
                padding-right: 2.5rem !important;
                border: 1px solid rgb(209 213 219);
                border-radius: 0.5rem;
                background-color: white;
                color: rgb(17 24 39);
                font-size: 0.875rem;
                appearance: none;
                -webkit-appearance: none;
                -moz-appearance: none;
                transition: all 0.2s;
            }

            .dark .form-select {
                border-color: rgb(75 85 99);
                background-color: rgb(55 65 81);
                color: white;
            }

            .form-select:focus {
                outline: none;
                ring: 2px;
                ring-color: rgb(59 130 246);
            }

            .action-btn-primary {
                height: 40px !important;
                min-height: 40px !important;
                max-height: 40px !important;
                box-sizing: border-box !important;
                padding: 0 1rem !important;
                margin: 0 !important;
                display: inline-flex !important;
                align-items: center !important;
                justify-content: center !important;
                font-weight: 500;
                border-radius: 0.5rem;
                transition: all 0.2s;
                white-space: nowrap;
                background-color: rgb(59 130 246);
                color: white;
                text-decoration: none;
                border: none;
                cursor: pointer;
            }

            .action-btn-primary:hover {
                background-color: rgb(37 99 235);
            }

            .action-btn-primary:disabled {
                opacity: 0.5;
                cursor: not-allowed;
            }

            .dark .action-btn-primary {
                background-color: rgb(59 130 246);
            }

            .dark .action-btn-primary:hover {
                background-color: rgb(37 99 235);
            }
        </style>
    @endsection
