// Real-time Notification System for Teknisi
class TeknisiNotificationSystem {
    constructor() {
        this.pollingInterval = 30000; // 30 seconds
        this.isPolling = false;
        this.isBadgeOpen = false;
        this.isToggling = false;
        this.init();
    }

    init() {
        console.log('[TeknisiNotificationSystem] Initializing...');
        this.startPolling();
        
        // Bind events after a short delay to ensure DOM is ready
        // This is important because navbar might not be rendered yet
        setTimeout(() => {
            console.log('[TeknisiNotificationSystem] Attempting to bind events...');
            this.bindEvents();
        }, 100);
        
        console.log('[TeknisiNotificationSystem] Initialized successfully');
    }

    startPolling() {
        if (this.isPolling) {
            console.log('[TeknisiNotificationSystem] Already polling, skipping...');
            return;
        }
        
        this.isPolling = true;
        console.log('[TeknisiNotificationSystem] Starting polling...');
        
        // Check immediately
        this.checkNotifications();
        
        // Then check every 30 seconds
        setInterval(() => {
            this.checkNotifications();
        }, this.pollingInterval);
        
        // Also check when page becomes visible
        document.addEventListener('visibilitychange', () => {
            if (!document.hidden) {
                console.log('[TeknisiNotificationSystem] Page visible, checking notifications...');
                this.checkNotifications();
            }
        });
    }

    async checkNotifications() {
        console.log('[TeknisiNotificationSystem] checkNotifications() called at:', new Date().toISOString());
        try {
            const csrfToken = document.querySelector('meta[name="csrf-token"]');
            if (!csrfToken) {
                console.error('[TeknisiNotificationSystem] CSRF token not found - cannot fetch notifications');
                return;
            }
            console.log('[TeknisiNotificationSystem] CSRF token found');

            const notificationUrl = '/teknisi/notifications';
            console.log('[TeknisiNotificationSystem] Fetching from URL:', notificationUrl);
            
            const response = await fetch(notificationUrl, {
                method: 'GET',
                headers: {
                    'X-CSRF-TOKEN': csrfToken.getAttribute('content'),
                    'Accept': 'application/json'
                }
            });

            console.log('[TeknisiNotificationSystem] Response received:', response.status, response.statusText);
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }

            const data = await response.json();
            
            console.log('[TeknisiNotificationSystem] ===== NOTIFICATION DATA RECEIVED =====');
            console.log('[TeknisiNotificationSystem] Data:', JSON.stringify(data, null, 2));
            
            if (data.success && data.data) {
                // Gunakan data langsung dari API tanpa filter localStorage
                // API sudah melakukan filtering berdasarkan is_read = false
                // Kita hanya perlu memastikan data yang ditampilkan sesuai dengan unread_counts
                let filteredData = data.data;
                    
                // Pastikan notifikasi yang ditampilkan hanya yang unread (is_read = false)
                // Ini untuk konsistensi antara badge count dan konten modal
                if (data.data.recent_items && data.data.recent_items.notifications) {
                    // Filter hanya notifikasi yang benar-benar unread
                    const unreadNotifications = data.data.recent_items.notifications.filter(notification => {
                        return !notification.is_read;
                    });
                    
                    // Update data dengan notifikasi unread saja
                    filteredData.recent_items.notifications = unreadNotifications;
                    
                    console.log('[TeknisiNotificationSystem] Filtered unread notifications:', {
                        original_count: data.data.recent_items.notifications.length,
                        unread_count: unreadNotifications.length,
                        unread_counts: filteredData.unread_counts
                    });
                }
                
                // SELALU update badges (navbar dan sidebar) meskipun modal belum terbuka
                // Gunakan unread_counts dari API (lebih akurat)
                this.updateBadge(data.data.unread_counts);
                
                // Update content jika modal sudah terbuka
                // Gunakan filteredData yang sudah di-filter untuk unread notifications
                    if (this.isBadgeOpen) {
                    this.updateNotificationContent(filteredData);
                }
            } else {
                console.error('[TeknisiNotificationSystem] Invalid response format:', data);
            }
        } catch (error) {
            console.error('[TeknisiNotificationSystem] Error fetching notifications:', error);
        }
    }

    updateBadge(unread_counts) {
        const badge = document.getElementById('notification-count');
        if (!badge) {
            console.warn('[TeknisiNotificationSystem] Badge element not found (ID: notification-count)');
            return;
        }

        const total = unread_counts?.total || 0;
        console.log('[TeknisiNotificationSystem] Updating badge with total:', total);

        if (total > 0) {
            badge.textContent = total > 99 ? '99+' : total;
            badge.classList.remove('hidden');
            badge.removeAttribute('hidden');
            badge.style.cssText = 'display: flex !important; visibility: visible !important;';
            console.log('[TeknisiNotificationSystem] Badge SHOWN with count:', total);
        } else {
            badge.classList.add('hidden');
            badge.setAttribute('hidden', 'true');
            badge.style.cssText = 'display: none !important; visibility: hidden !important;';
            console.log('[TeknisiNotificationSystem] Badge HIDDEN');
        }
        
        // Update sidebar productions badge
        const productionsBadge = document.getElementById('sidebar-productions-badge');
        if (productionsBadge) {
            const productionsCount = unread_counts?.productions || 0;
            productionsBadge.textContent = productionsCount > 99 ? '99+' : productionsCount.toString();
            if (productionsCount > 0) {
                productionsBadge.classList.remove('hidden');
                productionsBadge.removeAttribute('hidden');
                productionsBadge.style.cssText = 'display: inline-flex !important; visibility: visible !important; opacity: 1 !important;';
                console.log('[TeknisiNotificationSystem] Productions badge SHOWN:', productionsCount);
            } else {
                productionsBadge.classList.add('hidden');
                productionsBadge.style.cssText = 'display: none !important; visibility: hidden !important;';
                console.log('[TeknisiNotificationSystem] Productions badge HIDDEN');
            }
        } else {
            console.warn('[TeknisiNotificationSystem] Productions badge element not found (ID: sidebar-productions-badge)');
        }
        
        // Update sidebar quotations badge
        const quotationsBadge = document.getElementById('sidebar-quotations-badge');
        if (quotationsBadge) {
            const quotationsCount = unread_counts?.quotations || 0;
            console.log('[TeknisiNotificationSystem] Updating quotations badge with count:', quotationsCount);
            quotationsBadge.textContent = quotationsCount > 99 ? '99+' : quotationsCount.toString();
            if (quotationsCount > 0) {
                quotationsBadge.classList.remove('hidden');
                quotationsBadge.removeAttribute('hidden');
                quotationsBadge.style.cssText = 'display: inline-flex !important; visibility: visible !important; opacity: 1 !important;';
                console.log('[TeknisiNotificationSystem] Quotations badge SHOWN:', quotationsCount, 'Element:', quotationsBadge);
            } else {
                quotationsBadge.classList.add('hidden');
                quotationsBadge.setAttribute('hidden', 'true');
                quotationsBadge.style.cssText = 'display: none !important; visibility: hidden !important;';
                console.log('[TeknisiNotificationSystem] Quotations badge HIDDEN (count is 0)');
            }
        } else {
            console.warn('[TeknisiNotificationSystem] Quotations badge element not found (ID: sidebar-quotations-badge). Available badges:', {
                productions: !!document.getElementById('sidebar-productions-badge'),
                quotations: !!document.getElementById('sidebar-quotations-badge')
            });
        }
    }

    bindEvents() {
        console.log('[TeknisiNotificationSystem] bindEvents() called');
        console.log('[TeknisiNotificationSystem] Document readyState:', document.readyState);
        
        // Bind event ke tombol notifikasi dengan retry mechanism yang lebih robust (sama seperti admin/supervisor)
        const tryBind = () => {
            // Cari button dengan ID notification-toggle-btn atau teknisi-notification-btn
            const notificationBtn = document.getElementById('notification-toggle-btn') || document.getElementById('teknisi-notification-btn');
            console.log('[TeknisiNotificationSystem] Looking for button, found:', !!notificationBtn);
            
            if (notificationBtn) {
                console.log('[TeknisiNotificationSystem] Button found:', notificationBtn.id, notificationBtn.className);
                
                // Check if already bound (avoid duplicate listeners)
                if (notificationBtn.dataset.bound === 'true') {
                    console.log('[TeknisiNotificationSystem] Button already bound, skipping...');
                    return true;
                }
                
                console.log('[TeknisiNotificationSystem] Binding event listener to button...');
                
                // Remove existing listeners by cloning
                const newBtn = notificationBtn.cloneNode(true);
                notificationBtn.parentNode.replaceChild(newBtn, notificationBtn);
                
                // Mark as bound
                newBtn.dataset.bound = 'true';
                
                newBtn.addEventListener('click', (e) => {
                    e.preventDefault();
                    e.stopPropagation();
                    console.log('[TeknisiNotificationSystem] ===== NOTIFICATION BUTTON CLICKED =====');
                    console.log('[TeknisiNotificationSystem] window.notificationSystem exists:', !!window.notificationSystem);
                    if (window.notificationSystem) {
                        console.log('[TeknisiNotificationSystem] Calling toggleModal()...');
                        window.notificationSystem.toggleModal();
                    } else {
                        console.error('[TeknisiNotificationSystem] notificationSystem instance not found!');
                    }
                });
                console.log('[TeknisiNotificationSystem] ✓ Event bound to notification button successfully');
                return true;
            } else {
                console.warn('[TeknisiNotificationSystem] ✗ Notification button not found, will retry...');
                console.warn('[TeknisiNotificationSystem] Searched for: #notification-toggle-btn, #teknisi-notification-btn');
                return false;
            }
        };
        
        // Try immediately
        console.log('[TeknisiNotificationSystem] Attempting first bind...');
        if (!tryBind()) {
            console.log('[TeknisiNotificationSystem] First bind failed, retrying after 100ms...');
            // Retry after a short delay if button not found
            setTimeout(() => {
                if (!tryBind()) {
                    console.log('[TeknisiNotificationSystem] Second bind failed, retrying after 500ms...');
                    // Retry with longer delay
                    setTimeout(() => {
                        if (!tryBind()) {
                            console.log('[TeknisiNotificationSystem] Third bind failed, using MutationObserver...');
                            // Use MutationObserver to watch for button appearance
                            const observer = new MutationObserver((mutations, obs) => {
                                if (tryBind()) {
                                    console.log('[TeknisiNotificationSystem] MutationObserver found button and bound event');
                                    obs.disconnect();
                                }
                            });
                            
                            observer.observe(document.body, {
                                childList: true,
                                subtree: true
                            });
                            
                            // Stop observing after 5 seconds
                            setTimeout(() => {
                                observer.disconnect();
                                const btn1 = document.getElementById('notification-toggle-btn');
                                const btn2 = document.getElementById('teknisi-notification-btn');
                                if (!btn1 && !btn2) {
                                    console.error('[TeknisiNotificationSystem] ✗✗✗ Notification button never found after all retries!');
                                } else {
                                    console.log('[TeknisiNotificationSystem] Button exists but binding failed:', { btn1: !!btn1, btn2: !!btn2 });
                                }
                            }, 5000);
                        }
                    }, 500);
                }
            }, 100);
        }

        // Close modal when clicking outside
        if (!this._outsideClickHandler) {
            this._outsideClickHandler = (e) => {
                const modal = document.getElementById('notification-modal');
                const toggleBtn = e.target.closest('#notification-toggle-btn, #teknisi-notification-btn, .notification-icon');
                if (modal && this.isBadgeOpen && !modal.contains(e.target) && !toggleBtn) {
                    this.toggleModal();
                }
            };
            document.addEventListener('click', this._outsideClickHandler);
            console.log('[TeknisiNotificationSystem] Outside click handler bound');
        }
    }

    toggleModal() {
        if (this.isToggling) {
            console.log('[TeknisiNotificationSystem] Toggle in progress, ignoring...');
            return;
        }
        
        this.isToggling = true;
        
        let modal = document.getElementById('notification-modal');
        
        // Check actual modal state from DOM, not just from this.isBadgeOpen
        // This prevents state desync issues
        const isActuallyOpen = modal && modal.style.display !== 'none' && modal.style.visibility !== 'hidden' && !modal.classList.contains('hidden');
        const shouldOpen = !isActuallyOpen;
        
        console.log('[TeknisiNotificationSystem] Toggle called');
        console.log('[TeknisiNotificationSystem] this.isBadgeOpen:', this.isBadgeOpen);
        console.log('[TeknisiNotificationSystem] Modal exists:', !!modal);
        console.log('[TeknisiNotificationSystem] Modal actually open (from DOM):', isActuallyOpen);
        console.log('[TeknisiNotificationSystem] Should open:', shouldOpen);
        
        // Create modal jika belum ada
        if (!modal) {
            console.log('[TeknisiNotificationSystem] Creating new modal...');
            this.createNotificationModal();
            modal = document.getElementById('notification-modal');
        }
        
        if (!modal) {
            console.error('[TeknisiNotificationSystem] Modal element not found!');
            this.isToggling = false;
            return;
        }
        
        if (!shouldOpen) {
            // Tutup modal
            console.log('[TeknisiNotificationSystem] Closing modal...');
            this.isBadgeOpen = false;
            modal.style.opacity = '0';
            setTimeout(() => {
                if (modal) {
                    modal.style.display = 'none';
                    modal.style.visibility = 'hidden';
                }
                this.isToggling = false;
                console.log('[TeknisiNotificationSystem] Modal closed');
            }, 300);
        } else {
            // Buka modal
            console.log('[TeknisiNotificationSystem] Opening modal...');
            this.isBadgeOpen = true;
            
            // Set styles explicitly to ensure visibility
            modal.style.cssText = 'position: fixed; top: 72px; right: 1rem; width: 24rem; z-index: 9999; display: block; visibility: visible; opacity: 0;';
            
            // Force reflow
            void modal.offsetHeight;
            
            // Fade in
            requestAnimationFrame(() => {
                if (modal) {
                    modal.style.opacity = '1';
                    console.log('[TeknisiNotificationSystem] Modal opacity set to 1');
                    console.log('[TeknisiNotificationSystem] Modal computed styles:', {
                        display: window.getComputedStyle(modal).display,
                        visibility: window.getComputedStyle(modal).visibility,
                        opacity: window.getComputedStyle(modal).opacity,
                        zIndex: window.getComputedStyle(modal).zIndex
                    });
                }
                this.isToggling = false;
                console.log('[TeknisiNotificationSystem] Modal opened');
            });
            
            // Load notifications setelah modal terbuka
            setTimeout(() => {
                this.checkNotifications();
            }, 100);
        }
    }
    
    // Alias untuk backward compatibility
    toggleBadge() {
        this.toggleModal();
    }

    createNotificationModal() {
        // Remove existing modal if any (to prevent duplicates)
        const existingModal = document.getElementById('notification-modal');
        if (existingModal) {
            console.log('[TeknisiNotificationSystem] Removing existing modal...');
            existingModal.remove();
        }
        
        // Reset state
        this.isBadgeOpen = false;
        
        const modal = document.createElement('div');
        modal.id = 'notification-modal';
        modal.className = 'fixed right-4 z-50 w-full max-w-sm';
        // Position di bawah navbar - teknisi di dashboard, navbar lebih pendek (py-4 = 64px), jadi gunakan 72px
        modal.style.top = '72px';
        modal.style.display = 'none';
        modal.style.visibility = 'hidden';
        modal.style.opacity = '0';
        modal.style.transition = 'opacity 0.3s ease';
        
        modal.innerHTML = `
            <div class="bg-white rounded-xl shadow-2xl border border-gray-200 max-w-sm w-full max-h-[80vh] flex flex-col">
                <div class="p-4 border-b border-gray-200 flex-shrink-0">
                    <div class="flex items-center justify-between">
                        <h3 class="text-lg font-semibold text-gray-900">Notifikasi</h3>
                        <div class="flex items-center space-x-2">
                            <button id="teknisi-mark-all-read-btn" 
                                class="text-xs bg-blue-600 hover:bg-blue-700 text-white px-3 py-1.5 rounded transition-colors cursor-pointer">
                                Tandai Semua Dibaca
                            </button>
                            <button onclick="if(window.notificationSystem){window.notificationSystem.toggleModal();}" 
                                    class="text-gray-400 hover:text-gray-600 transition-colors p-1 rounded hover:bg-gray-100"
                                title="Tutup">
                                <i class="fas fa-times text-xl"></i>
                            </button>
                        </div>
                    </div>
                </div>
                <div id="notification-content" class="flex-1 overflow-y-auto p-4" style="max-height: calc(80vh - 120px);">
                    <div class="text-center py-8">
                        <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
                        <p class="text-gray-500 mt-2">Memuat notifikasi...</p>
                    </div>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        console.log('[TeknisiNotificationSystem] Modal created and added to DOM');
        console.log('[TeknisiNotificationSystem] Modal initial state - display:', modal.style.display, 'visibility:', modal.style.visibility);
        
        // Bind event untuk tombol "Tandai Semua Dibaca"
        const markAllReadBtn = document.getElementById('teknisi-mark-all-read-btn');
        if (markAllReadBtn) {
            markAllReadBtn.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                console.log('[TeknisiNotificationSystem] Mark all read button clicked');
                if (this.markAllAsRead) {
                    this.markAllAsRead();
                } else {
                    console.error('[TeknisiNotificationSystem] markAllAsRead method not found');
                }
            });
            console.log('[TeknisiNotificationSystem] Mark all read button event bound');
        } else {
            console.warn('[TeknisiNotificationSystem] Mark all read button not found');
        }
    }

    updateNotificationContent(data) {
        const content = document.getElementById('notification-content');
        if (!content) {
            console.error('[TeknisiNotificationSystem] Content element not found');
            return;
        }

        try {
            const { unread_counts, recent_items } = data;
            
            console.log('[TeknisiNotificationSystem] Updating content:', {
                total: unread_counts.total,
                inquiries: unread_counts.inquiries,
                productions: unread_counts.productions,
                recent_notifications: recent_items.notifications?.length || 0,
                recent_notifications_data: recent_items.notifications || []
            });
            
            if (unread_counts.total === 0) {
                content.innerHTML = `
                    <div class="text-center text-gray-500 dark:text-gray-400 py-8">
                        <i class="fas fa-check-circle text-green-500 text-3xl mb-3"></i>
                        <p class="font-medium">Tidak ada notifikasi baru</p>
                    </div>
                `;
                return;
            }
            
            let html = '<div class="space-y-3">';

            // TAMPILKAN NOTIFIKASI DARI recent_items.notifications
            const notifications = recent_items.notifications || [];
            
            // Pisahkan notifikasi berdasarkan type
            const productionNotifications = notifications.filter(n => n.type === 'production');
            const inquiryNotifications = notifications.filter(n => n.type === 'inquiry');

            // TAMPILKAN NOTIFIKASI PRODUKSI TERLEBIH DAHULU (PRIORITAS TINGGI)
            if (productionNotifications.length > 0 || (unread_counts.productions || 0) > 0) {
                html += '<div class="text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Notifikasi Produksi</div>';
                
                if (productionNotifications.length > 0) {
                    productionNotifications.forEach(notification => {
                        const production = notification.production;
                        if (!production) return;
                        const productionUrl = `/teknisi/productions/${production.id}`;
                        const order = production.order || {};
                        const product = production.product || {};
                        
                        // Tentukan warna berdasarkan subtype notification
                        let cardBgColor = 'bg-indigo-50 dark:bg-indigo-900/20';
                        let cardBorderColor = 'border-indigo-500';
                        let cardHoverColor = 'hover:bg-indigo-100 dark:hover:bg-indigo-900/30';
                        let textColor = 'text-indigo-900 dark:text-indigo-100';
                        let iconHtml = '<i class="fas fa-industry mr-1"></i>';
                        
                        // Tentukan warna berdasarkan subtype
                        if (notification.subtype === 'production_assigned') {
                            cardBgColor = 'bg-blue-50 dark:bg-blue-900/20';
                            cardBorderColor = 'border-blue-500';
                            cardHoverColor = 'hover:bg-blue-100 dark:hover:bg-blue-900/30';
                            textColor = 'text-blue-900 dark:text-blue-100';
                            iconHtml = '<i class="fas fa-user-plus mr-1"></i>';
                        } else if (notification.subtype === 'planning_approved') {
                            cardBgColor = 'bg-green-50 dark:bg-green-900/20';
                            cardBorderColor = 'border-green-500';
                            cardHoverColor = 'hover:bg-green-100 dark:hover:bg-green-900/30';
                            textColor = 'text-green-900 dark:text-green-100';
                            iconHtml = '<i class="fas fa-check-circle mr-1"></i>';
                        } else if (notification.subtype === 'planning_rejected') {
                            cardBgColor = 'bg-red-50 dark:bg-red-900/20';
                            cardBorderColor = 'border-red-500';
                            cardHoverColor = 'hover:bg-red-100 dark:hover:bg-red-900/30';
                            textColor = 'text-red-900 dark:text-red-100';
                            iconHtml = '<i class="fas fa-times-circle mr-1"></i>';
                        } else if (notification.subtype === 'completion_approved') {
                            cardBgColor = 'bg-emerald-50 dark:bg-emerald-900/20';
                                cardBorderColor = 'border-emerald-500';
                            cardHoverColor = 'hover:bg-emerald-100 dark:hover:bg-emerald-900/30';
                            textColor = 'text-emerald-900 dark:text-emerald-100';
                                iconHtml = '<i class="fas fa-check-double mr-1"></i>';
                        } else if (notification.subtype === 'completion_rejected') {
                            cardBgColor = 'bg-red-50 dark:bg-red-900/20';
                            cardBorderColor = 'border-red-500';
                            cardHoverColor = 'hover:bg-red-100 dark:hover:bg-red-900/30';
                            textColor = 'text-red-900 dark:text-red-100';
                            iconHtml = '<i class="fas fa-times-circle mr-1"></i>';
                        }
                        
                        html += `
                            <a href="${productionUrl}" 
                               data-notification-id="${notification.id}"
                               class="notification-item block p-3 ${cardBgColor} rounded-lg border-l-4 ${cardBorderColor} ${cardHoverColor} transition-colors cursor-pointer"
                               style="text-decoration: none; color: inherit;">
                                <div class="flex items-start justify-between">
                                    <div class="flex-1">
                                        <div class="flex items-center gap-2 mb-1">
                                            <p class="font-medium ${textColor}">${order.customer_name || notification.data?.customer_name || 'N/A'}</p>
                                            <span class="px-2 py-0.5 bg-white dark:bg-gray-700 ${textColor} text-xs rounded-full font-semibold">#${production.order_id || production.id}</span>
                                        </div>
                                        <p class="text-sm ${textColor.replace('900', '700').replace('100', '300')} mt-1">
                                            ${iconHtml}${notification.message}
                                        </p>
                                        <p class="text-xs ${textColor.replace('900', '600').replace('100', '400')} mt-1">
                                            <i class="fas fa-box mr-1"></i>${product.product_title || notification.data?.product_title || 'Produk tidak ditemukan'}
                                        </p>
                                        <p class="text-xs ${textColor.replace('900', '600').replace('100', '400')} mt-1">
                                            <i class="fas fa-clock mr-1"></i>${this.formatTime(notification.created_at)}
                                        </p>
                                    </div>
                                    <div class="${textColor.replace('900', '600').replace('100', '400')} hover:opacity-80 ml-2">
                                        <i class="fas fa-external-link-alt"></i>
                                    </div>
                                </div>
                            </a>
                        `;
                    });
                } else if ((unread_counts.productions || 0) > 0) {
                    html += `
                        <a href="/teknisi/productions" 
                           class="block p-3 bg-indigo-50 dark:bg-indigo-900/20 rounded-lg border-l-4 border-indigo-500 hover:bg-indigo-100 dark:hover:bg-indigo-900/30 transition-colors cursor-pointer"
                           style="text-decoration: none; color: inherit;">
                            <div class="flex items-start justify-between">
                                <div class="flex-1">
                                    <p class="font-medium text-indigo-900 dark:text-indigo-100 mb-1">Ada ${unread_counts.productions} notifikasi produksi</p>
                                    <p class="text-sm text-indigo-700 dark:text-indigo-300 mt-1">Ada update pada produksi Anda</p>
                                </div>
                                <div class="text-indigo-600 dark:text-indigo-400 hover:text-indigo-800 dark:hover:text-indigo-200 ml-2">
                                    <i class="fas fa-external-link-alt"></i>
                                </div>
                            </div>
                        </a>
                    `;
                }
            }

            // TAMPILKAN PENAWARAN YANG PERLU ESTIMASI
            if (inquiryNotifications.length > 0 || (unread_counts.inquiries || 0) > 0) {
                html += '<div class="text-sm font-medium text-gray-700 dark:text-gray-300 mb-2 mt-4">Penawaran Perlu Estimasi</div>';
                
                if (inquiryNotifications.length > 0) {
                    inquiryNotifications.forEach(notification => {
                        const inquiry = notification.inquiry;
                        if (!inquiry) return;
                        
                        const inquiryUrl = `/teknisi/quotations?search=${encodeURIComponent(inquiry.id)}`;
                        
                        html += `
                            <a href="${inquiryUrl}" 
                               data-notification-id="${notification.id}"
                               class="notification-item block p-3 bg-orange-50 dark:bg-orange-900/20 rounded-lg border-l-4 border-orange-500 hover:bg-orange-100 dark:hover:bg-orange-900/30 transition-colors cursor-pointer"
                               style="text-decoration: none; color: inherit;">
                                <div class="flex items-start justify-between">
                                    <div class="flex-1">
                                        <div class="flex items-center gap-2 mb-1">
                                            <p class="font-medium text-orange-900 dark:text-orange-100">${inquiry.name || 'N/A'}</p>
                                            <span class="px-2 py-0.5 bg-orange-100 dark:bg-orange-800 text-orange-700 dark:text-orange-200 text-xs rounded-full font-semibold">Estimasi</span>
                                        </div>
                                        <p class="text-sm text-orange-700 dark:text-orange-300 mt-1">
                                            <i class="fas fa-tools mr-1"></i>${notification.message}
                                        </p>
                                        <p class="text-xs text-orange-600 dark:text-orange-400 mt-1">
                                            <i class="fas fa-clock mr-1"></i>${this.formatTime(notification.created_at)}
                                        </p>
                                    </div>
                                    <div class="text-orange-600 dark:text-orange-400 hover:text-orange-800 dark:hover:text-orange-200 ml-2">
                                        <i class="fas fa-external-link-alt"></i>
                                    </div>
                                </div>
                            </a>
                        `;
                    });
                } else if ((unread_counts.inquiries || 0) > 0) {
                    html += `
                        <a href="/teknisi/quotations" 
                           class="block p-3 bg-orange-50 dark:bg-orange-900/20 rounded-lg border-l-4 border-orange-500 hover:bg-orange-100 dark:hover:bg-orange-900/30 transition-colors cursor-pointer"
                           style="text-decoration: none; color: inherit;">
                            <div class="flex items-start justify-between">
                                <div class="flex-1">
                                    <p class="font-medium text-orange-900 dark:text-orange-100 mb-1">Ada ${unread_counts.inquiries || 0} penawaran yang perlu estimasi</p>
                                </div>
                                <div class="text-orange-600 dark:text-orange-400 hover:text-orange-800 dark:hover:text-orange-200 ml-2">
                                    <i class="fas fa-external-link-alt"></i>
                                </div>
                            </div>
                        </a>
                    `;
                }
            }

            // Jika tidak ada notifikasi sama sekali
            if (html === '<div class="space-y-3">') {
                html += `
                    <div class="text-center py-8">
                        <i class="fas fa-bell-slash text-gray-400 text-4xl mb-4"></i>
                        <p class="text-gray-500 dark:text-gray-400">Tidak ada notifikasi baru</p>
                    </div>
                `;
            }

            html += '</div>';
            
            // Footer
            html += `
                <div class="mt-4 pt-4 border-t border-gray-200 dark:border-gray-700 text-xs text-gray-500 dark:text-gray-400 flex justify-between">
                    <span>Total: ${unread_counts.total} notifikasi baru</span>
                    <span>Terakhir update: ${this.formatTime(data.last_updated)}</span>
                </div>
            `;

            content.innerHTML = html;
            
            // Bind click events untuk semua link
            this.bindNotificationClicks();
            
        } catch (error) {
            console.error('[TeknisiNotificationSystem] Error updating content:', error);
            content.innerHTML = `
                <div class="text-center text-red-500 py-8">
                    <i class="fas fa-exclamation-circle text-3xl mb-3"></i>
                    <p class="font-medium">Error memuat notifikasi</p>
                    <p class="text-sm mt-2">${error.message}</p>
                </div>
            `;
        }
    }

    bindNotificationClicks() {
        // Bind click events untuk semua link di notification content
        const notificationItems = document.querySelectorAll('#notification-content .notification-item');
        notificationItems.forEach(item => {
            item.addEventListener('click', async (e) => {
                const notificationId = item.dataset.notificationId;
                if (notificationId) {
                    // Mark as read secara async (tidak blocking navigation)
                    this.markNotificationAsRead(notificationId).catch(err => {
                        console.error('[TeknisiNotificationSystem] Error marking notification as read:', err);
                    });
                }
                // Tutup modal setelah klik
                setTimeout(() => {
                    this.toggleModal();
                }, 200);
        });
        });
    }

    /**
     * Mark single notification as read
     */
    async markNotificationAsRead(notificationId) {
        try {
            const csrfToken = document.querySelector('meta[name="csrf-token"]');
            if (!csrfToken) return;
            
            const response = await fetch(`/teknisi/notifications/${notificationId}/mark-read`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken.getAttribute('content'),
                    'Accept': 'application/json'
                }
            });

            if (!response.ok) {
                throw new Error(`Network response was not ok: ${response.status}`);
            }

            const data = await response.json();
            console.log('[TeknisiNotificationSystem] Mark as read response:', data);
            
            if (data.success && data.data && data.data.unread_counts) {
                // Update badge dengan unread_counts yang baru
                this.updateBadge(data.data.unread_counts);
            }
        } catch (error) {
            console.error('[TeknisiNotificationSystem] Error marking notification as read:', error);
        }
    }

    formatTime(timeString) {
        if (!timeString) return 'Tidak diketahui';
        
        try {
            const time = new Date(timeString);
            const now = new Date();
            const diffMs = now - time;
            const diffSecs = Math.floor(diffMs / 1000);
            const diffMins = Math.floor(diffSecs / 60);
            const diffHours = Math.floor(diffMins / 60);
            const diffDays = Math.floor(diffHours / 24);
            
            if (diffSecs < 60) {
                return 'Baru saja';
            } else if (diffMins < 60) {
                return `${diffMins} menit yang lalu`;
            } else if (diffHours < 24) {
                return `${diffHours} jam yang lalu`;
            } else if (diffDays < 7) {
                return `${diffDays} hari yang lalu`;
            } else {
                return time.toLocaleDateString('id-ID', {
                    day: 'numeric',
                    month: 'short',
                    year: 'numeric',
                    hour: '2-digit',
                    minute: '2-digit'
                });
            }
        } catch (error) {
            console.error('[TeknisiNotificationSystem] Error formatting time:', error);
            return 'Tidak diketahui';
        }
    }

    /**
     * Mark all notifications as read
     */
    async markAllAsRead() {
        try {
            console.log('[TeknisiNotificationSystem] Marking all as read...');
            
            const csrfToken = document.querySelector('meta[name="csrf-token"]');
            if (!csrfToken) {
                console.error('[TeknisiNotificationSystem] CSRF token not found');
                return;
            }
            
            const response = await fetch('/teknisi/notifications/mark-all-read', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken.getAttribute('content'),
                    'Accept': 'application/json'
                }
            });

            if (!response.ok) {
                const errorText = await response.text();
                console.error('[TeknisiNotificationSystem] Response not OK:', response.status, errorText);
                throw new Error(`Network response was not ok: ${response.status}`);
            }

            const data = await response.json();
            console.log('[TeknisiNotificationSystem] Mark all read response:', data);
            
            if (data.success) {
                // Langsung update badge menjadi 0
                this.updateBadge({ total: 0, productions: 0, inquiries: 0 });
                
                // Update content untuk menampilkan "Tidak ada notifikasi baru"
                const content = document.getElementById('notification-content');
                if (content) {
                    content.innerHTML = `
                        <div class="text-center py-8">
                            <i class="fas fa-check-circle text-green-500 text-3xl mb-3"></i>
                            <p class="font-medium text-gray-700 dark:text-gray-300">Tidak ada notifikasi baru</p>
                            <p class="text-sm text-gray-500 dark:text-gray-400 mt-2">Semua notifikasi telah ditandai sebagai dibaca</p>
                        </div>
                    `;
                }
                
                // Tampilkan notifikasi sukses
                if (typeof Swal !== 'undefined') {
                    Swal.fire({
                        title: 'Berhasil!',
                        text: 'Semua notifikasi telah ditandai sebagai dibaca',
                        icon: 'success',
                        iconColor: '#10b981',
                        confirmButtonColor: '#3b82f6',
                        confirmButtonText: 'OK',
                        timer: 2000,
                        timerProgressBar: true,
                        allowOutsideClick: false,
                        allowEscapeKey: false,
                        showClass: {
                            popup: 'animate__animated animate__fadeInDown animate__faster'
                        },
                        hideClass: {
                            popup: 'animate__animated animate__fadeOutUp animate__faster'
                        }
                    });
                }
                
                // Refresh notifications setelah 1 detik untuk update dari server
                setTimeout(() => {
                    console.log('[TeknisiNotificationSystem] Refreshing notifications after mark all read...');
                    this.checkNotifications();
                }, 1000);
            } else {
                console.error('[TeknisiNotificationSystem] Mark all read failed:', data.message);
                if (typeof Swal !== 'undefined') {
                    Swal.fire({
                        title: 'Gagal!',
                        text: data.message || 'Gagal menandai semua notifikasi sebagai dibaca',
                        icon: 'error',
                        confirmButtonColor: '#ef4444',
                        confirmButtonText: 'OK'
                    });
                }
            }
        } catch (error) {
            console.error('[TeknisiNotificationSystem] Error marking all notifications as read:', error);
            if (typeof Swal !== 'undefined') {
                Swal.fire({
                    title: 'Error!',
                    text: 'Terjadi kesalahan saat menandai notifikasi: ' + error.message,
                    icon: 'error',
                    confirmButtonColor: '#ef4444',
                    confirmButtonText: 'OK'
                });
            }
        }
    }
}

// Initialize notification system (same pattern as admin/supervisor notifications.js)
console.log('[TeknisiNotificationSystem] Script loaded, document readyState:', document.readyState);
console.log('[TeknisiNotificationSystem] Current path:', window.location.pathname);

if (document.readyState === 'loading') {
    console.log('[TeknisiNotificationSystem] DOM still loading, waiting for DOMContentLoaded...');
    document.addEventListener('DOMContentLoaded', function() {
        console.log('[TeknisiNotificationSystem] DOMContentLoaded fired, initializing...');
        if (!window.notificationSystem) {
            console.log('[TeknisiNotificationSystem] Creating new TeknisiNotificationSystem instance...');
            window.notificationSystem = new TeknisiNotificationSystem();
        } else {
            console.log('[TeknisiNotificationSystem] NotificationSystem already exists, re-binding events...');
            // Re-bind events if system already exists (e.g., if script was loaded twice)
            if (window.notificationSystem && typeof window.notificationSystem.bindEvents === 'function') {
                setTimeout(() => {
                    window.notificationSystem.bindEvents();
                }, 200);
            }
        }
    });
} else {
    console.log('[TeknisiNotificationSystem] DOM already ready, initializing immediately...');
    if (!window.notificationSystem) {
        console.log('[TeknisiNotificationSystem] Creating new TeknisiNotificationSystem instance...');
        window.notificationSystem = new TeknisiNotificationSystem();
    } else {
        console.log('[TeknisiNotificationSystem] NotificationSystem already exists, re-binding events...');
        // Re-bind events if system already exists
        if (window.notificationSystem && typeof window.notificationSystem.bindEvents === 'function') {
            setTimeout(() => {
                window.notificationSystem.bindEvents();
            }, 200);
        }
    }
}

// Also try to bind events after window load (as a final fallback)
window.addEventListener('load', function() {
    console.log('[TeknisiNotificationSystem] Window loaded, checking if events need binding...');
    if (window.notificationSystem && typeof window.notificationSystem.bindEvents === 'function') {
        const btn = document.getElementById('notification-toggle-btn') || document.getElementById('teknisi-notification-btn');
        if (btn && btn.dataset.bound !== 'true') {
            console.log('[TeknisiNotificationSystem] Button found but not bound, binding now...');
            setTimeout(() => {
                window.notificationSystem.bindEvents();
            }, 100);
        } else if (btn) {
            console.log('[TeknisiNotificationSystem] Button already bound');
        } else {
            console.warn('[TeknisiNotificationSystem] Button not found even after window load');
        }
    }
});

