<?php

namespace App\Providers;

use App\Models\Category;
use App\Models\Setting;
use Illuminate\Pagination\Paginator;
use Illuminate\Support\ServiceProvider;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        // Set default pagination view ke Tailwind
        Paginator::defaultView('vendor.pagination.tailwind');
        Paginator::defaultSimpleView('vendor.pagination.simple-tailwind');
        
        // Override Midtrans config dengan database settings
        // Prioritas: Database Settings > .env
        // Jika value di database kosong/empty, fallback ke .env
        try {
            $paymentSettings = Setting::getByGroup('payment');
            
            $serverKey = !empty($paymentSettings['midtrans_server_key']) 
                ? $paymentSettings['midtrans_server_key'] 
                : config('midtrans.server_key');
            
            $clientKey = !empty($paymentSettings['midtrans_client_key']) 
                ? $paymentSettings['midtrans_client_key'] 
                : config('midtrans.client_key');
            
            $isProduction = isset($paymentSettings['midtrans_is_production']) 
                ? ($paymentSettings['midtrans_is_production'] === '1') 
                : config('midtrans.is_production', false);
            
            config([
                'midtrans.server_key' => $serverKey,
                'midtrans.client_key' => $clientKey,
                'midtrans.is_production' => $isProduction,
            ]);
        } catch (\Exception $e) {
            // Jika database belum siap atau terjadi error, gunakan config default dari .env
            // Error akan di-log tapi tidak akan menghentikan aplikasi
        }
        
        view()->composer('partials.navbar', function ($view) {
            // Ambil kategori besar (parent) untuk navbar dengan sub kategori dan produk
            $view->with('productCategories', Category::where('type', 'service')
                ->whereNull('parent_id') // Hanya kategori besar
                ->with(['children' => function($query) {
                    $query->orderBy('name');
                }, 'products' => function($query) {
                    $query->where('is_active', true)
                          ->with('images')
                          ->latest('id')
                          ->take(6); // Ambil 6 produk terbaru per kategori
                }])
                ->orderBy('name')
                ->take(8) // ambil hanya 8 kategori besar
                ->get());
        });

        // Share company settings ke header dan footer
        view()->composer(['partials.header', 'partials.footer'], function ($view) {
            $companySettings = Setting::getByGroup('company');
            $view->with([
                'companyName' => $companySettings['company_name'] ?? 'POLJAM TECH',
                'companyLogo' => $companySettings['company_logo'] ?? null,
                'companyDescription' => $companySettings['company_description'] ?? 'Menyediakan Mesin Teknologi Terdepan untuk Bisnis Anda',
                'companyEmail' => $companySettings['company_email'] ?? 'info@poljamtech.com',
                'companyPhone' => $companySettings['company_phone'] ?? '+62 XXX XXX XXXX',
                'companyWebsite' => $companySettings['company_website'] ?? null,
                'companyAddress' => $companySettings['company_address'] ?? 'Indonesia',
            ]);
        });
    }
}
