<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Sparepart extends Model
{
    use HasFactory;

    protected $fillable = [
        'part_code',
        'name',
        'brand',
        'type',
        'part_number',
        'stock',
        'damaged_stock',
        'unit',
        'price',
        'image',
        'description',
        'is_active',
        'condition',
        'damage_reason',
        'damage_date',
        'damage_notes',
    ];

    protected $casts = [
        'stock' => 'integer',
        'damaged_stock' => 'integer',
        'price' => 'decimal:2',
        'is_active' => 'boolean',
        'damage_date' => 'datetime',
    ];

    /**
     * Accessor untuk mendapatkan stock yang baik (total - rusak)
     */
    public function getGoodStockAttribute()
    {
        return max(0, $this->stock - $this->damaged_stock);
    }

    /**
     * Accessor untuk mendapatkan persentase kerusakan
     */
    public function getDamagePercentageAttribute()
    {
        if ($this->stock == 0) return 0;
        return round(($this->damaged_stock / $this->stock) * 100, 2);
    }

    /**
     * Scope untuk sparepart dengan kerusakan
     */
    public function scopeHasDamage($query)
    {
        return $query->where('damaged_stock', '>', 0);
    }

    /**
     * Scope untuk sparepart dengan kerusakan tinggi (>50%)
     */
    public function scopeHighDamage($query)
    {
        return $query->whereRaw('(damaged_stock / stock) > 0.5');
    }

    // Relasi ke produk sudah tidak digunakan karena sekarang menggunakan input bebas
    // Relasi ini dihapus karena foreign key sparepart_id sudah dihapus dari tabel product_sparepart
    // /**
    //  * Relasi ke produk
    //  */
    // public function products()
    // {
    //     return $this->belongsToMany(Product::class, 'product_sparepart')
    //         ->withPivot(['quantity', 'unit', 'usage_cost'])
    //         ->withTimestamps();
    // }

    /**
     * Relasi ke produksi
     */
    public function productions()
    {
        return $this->belongsToMany(Production::class, 'production_spareparts')
            ->withPivot(['quantity', 'unit', 'unit_cost', 'total_cost'])
            ->withTimestamps();
    }

    /**
     * relasi ke pembelian sparepart
     */
    public function purchaseItems()
    {
        return $this->hasMany(\App\Models\PurchaseItem::class, 'item_id')
            ->where('item_type', 'sparepart');
    }

    /**
     * Relasi ke stock opname
     */
    public function stockOpnameItems()
    {
        return $this->morphMany(StockOpnameItem::class, 'itemable');
    }

    /**
     * Scope untuk barang dengan stock rendah
     */
    public function scopeLowStock($query, $threshold = 5)
    {
        return $query->where('stock', '<=', $threshold);
    }

    /**
     * Scope untuk barang aktif
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Accessor untuk status stock
     */
    public function getStockStatusAttribute()
    {
        if ($this->stock <= 3) {
            return 'critical';
        } elseif ($this->stock <= 5) {
            return 'low';
        } elseif ($this->stock <= 10) {
            return 'medium';
        } else {
            return 'good';
        }
    }

    /**
     * Accessor untuk label status stock
     */
    public function getStockStatusLabelAttribute()
    {
        return match($this->stock_status) {
            'critical' => 'Kritis',
            'low' => 'Rendah',
            'medium' => 'Sedang',
            'good' => 'Baik',
            default => 'Tidak Diketahui'
        };
    }
}
