<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Production extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_id',
        'product_id',
        'kd_project',
        'teknisi_id',
        'service_package_id', // paket service yang dipilih untuk upah jasa
        'quantity',
        'start_date',
        'end_date',
        'status', // menunggu, dalam_proses, selesai, dibatalkan
        'planning_status', // planning, pending_approval, approved, rejected
        'completion_status', // pending_approval, approved, rejected
        'materials_status', // pending, preparing, received, rejected
        'spareparts_status', // pending, preparing, received, rejected
        'total_material_cost',
        'total_sparepart_cost',
        'total_production_cost',
        'labor_cost', // upah pengerjaan per paket
        'labor_package_count', // jumlah paket untuk perhitungan upah (bisa berbeda dari quantity)
        'notes',
        'supervisor_feedback',
        'assigned_at',
        'submitted_at',
        'approved_at',
        'approved_by',
        'completed_at',
        'completion_approved_at',
        'completion_approved_by',
        'materials_received_at',
        'materials_received_by',
        'spareparts_received_at',
        'spareparts_received_by',
        'estimated_duration_days',
        'material_preparation_days',
        'actual_start_date',
        'estimated_completion_date',
        // Field untuk hasil produksi
        'result_photos',
        'test_results',
        'final_specifications',
        'completion_notes',
    ];

    protected $casts = [
        'quantity' => 'integer',
        'start_date' => 'date',
        'end_date' => 'date',
        'actual_start_date' => 'date',
        'estimated_completion_date' => 'date',
        'total_material_cost' => 'decimal:2',
        'total_sparepart_cost' => 'decimal:2',
        'total_production_cost' => 'decimal:2',
        'labor_cost' => 'decimal:2',
        'labor_package_count' => 'integer',
        'assigned_at' => 'datetime',
        'submitted_at' => 'datetime',
        'approved_at' => 'datetime',
        'completed_at' => 'datetime',
        'completion_approved_at' => 'datetime',
        'materials_received_at' => 'datetime',
        'spareparts_received_at' => 'datetime',
        'estimated_duration_days' => 'integer',
        'material_preparation_days' => 'integer',
        'result_photos' => 'array',
    ];

    /*
    |--------------------------------------------------------------------------
    | RELATIONS
    |--------------------------------------------------------------------------
    */
    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function teknisi()
    {
        return $this->belongsTo(User::class, 'teknisi_id');
    }

    public function supervisor()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function completionApprover()
    {
        return $this->belongsTo(User::class, 'completion_approved_by');
    }

    public function materialsReceiver()
    {
        return $this->belongsTo(User::class, 'materials_received_by');
    }

    public function sparepartsReceiver()
    {
        return $this->belongsTo(User::class, 'spareparts_received_by');
    }

    public function servicePackage()
    {
        return $this->belongsTo(ServicePackage::class);
    }

    public function materials()
    {
        return $this->belongsToMany(Material::class, 'production_materials')
            ->withPivot(['quantity', 'unit', 'unit_cost', 'total_cost'])
            ->withTimestamps();
    }

    public function spareparts()
    {
        return $this->belongsToMany(Sparepart::class, 'production_spareparts')
            ->withPivot(['quantity', 'unit', 'unit_cost', 'total_cost'])
            ->withTimestamps();
    }

    public function productionMaterials()
    {
        return $this->hasMany(ProductionMaterial::class);
    }

    public function productionSpareparts()
    {
        return $this->hasMany(ProductionSparepart::class);
    }

    public function itemRequests()
    {
        return $this->hasMany(ProductionItemRequest::class);
    }

    public function pendingItemRequests()
    {
        return $this->hasMany(ProductionItemRequest::class)->where('status', 'pending');
    }

    /*
    |--------------------------------------------------------------------------
    | COST CALCULATION
    |--------------------------------------------------------------------------
    */
    public function calculateCosts()
    {
        $materialTotal = $this->productionMaterials()->sum('total_cost') ?? 0;
        $sparepartTotal = $this->productionSpareparts()->sum('total_cost') ?? 0;

        $this->updateQuietly([
            'total_material_cost' => $materialTotal,
            'total_sparepart_cost' => $sparepartTotal,
            'total_production_cost' => $materialTotal + $sparepartTotal,
        ]);
    }

    /**
     * Total biaya langsung (bahan + sparepart)
     */
    public function getTotalDirectCostAttribute()
    {
        $materials = $this->total_material_cost ?? 0;
        $spareparts = $this->total_sparepart_cost ?? 0;
        return $materials + $spareparts;
    }

    /**
     * Total biaya aktual (bahan + sparepart + labor)
     */
    public function getTotalActualCostAttribute()
    {
        // Total upah = upah per paket × jumlah paket upah
        $laborPerPackage = $this->labor_cost ?? 0;
        // Handle null dengan aman menggunakan isset
        $packageCount = (isset($this->attributes['labor_package_count']) && $this->attributes['labor_package_count'] !== null)
            ? (int)$this->attributes['labor_package_count']
            : ($this->quantity ?? 1);
        $laborTotal = $laborPerPackage * $packageCount;
        return $this->total_direct_cost + $laborTotal;
    }

    /**
     * Biaya langsung per unit
     */
    public function getTotalDirectCostPerUnitAttribute()
    {
        $qty = $this->order->quantity ?? $this->quantity ?? 1;
        $qty = max(1, (int) $qty);
        return $this->total_direct_cost / $qty;
    }

    /**
     * Biaya aktual per unit
     */
    public function getTotalActualCostPerUnitAttribute()
    {
        $qty = $this->order->quantity ?? $this->quantity ?? 1;
        $qty = max(1, (int) $qty);
        return $this->total_actual_cost / $qty;
    }

    /**
     * Total upah jasa (upah per paket × jumlah paket)
     */
    public function getTotalLaborCostAttribute()
    {
        $laborPerPackage = $this->labor_cost ?? 0;
        // Handle null dengan aman menggunakan isset
        $packageCount = (isset($this->attributes['labor_package_count']) && $this->attributes['labor_package_count'] !== null)
            ? (int)$this->attributes['labor_package_count']
            : ($this->quantity ?? 1);
        return $laborPerPackage * $packageCount;
    }

    /**
     * Upah (labor) per unit produk
     */
    public function getLaborPerUnitAttribute()
    {
        $qty = $this->order->quantity ?? $this->quantity ?? 1;
        $qty = max(1, (int) $qty);
        $totalLabor = $this->total_labor_cost;
        return $totalLabor / $qty;
    }

    /**
     * Get calculated total production cost from materials and spareparts
     */
    public function getCalculatedProductionCostAttribute()
    {
        $materialTotal = $this->productionMaterials()->sum('total_cost') ?? 0;
        $sparepartTotal = $this->productionSpareparts()->sum('total_cost') ?? 0;
        return $materialTotal + $sparepartTotal;
    }

    /**
     * Get calculated material cost from materials
     */
    public function getCalculatedMaterialCostAttribute()
    {
        return $this->productionMaterials()->sum('total_cost') ?? 0;
    }

    /**
     * Get calculated sparepart cost from spareparts
     */
    public function getCalculatedSparepartCostAttribute()
    {
        return $this->productionSpareparts()->sum('total_cost') ?? 0;
    }

    /*
    |--------------------------------------------------------------------------
    | STATUS SYNC WITH ORDER
    |--------------------------------------------------------------------------
    */
    protected static function booted()
    {
        // Saat produksi baru dibuat → ubah status order ke "menunggu" + set estimasi
        static::created(function ($production) {
            if ($production->order && $production->status === 'menunggu') {
                $production->order->updateQuietly([
                    'production_status' => 'menunggu'
                ]);
            }

            // Set estimated duration dari product jika ada
            if ($production->product && $production->product->product_work_duration) {
                // Convert string duration to integer days
                $durationDays = static::parseDurationToDays($production->product->product_work_duration);
                $production->updateQuietly([
                    'estimated_duration_days' => $durationDays,
                    'materials_status' => 'pending',
                    'material_preparation_days' => 2, // default 2 hari
                ]);
            }

            // Set labor_cost dari product.service_fee sebagai default
            if ($production->product && $production->product->service_fee && !$production->labor_cost) {
                $production->updateQuietly([
                    'labor_cost' => $production->product->service_fee,
                ]);
            }
        });

        // Saat status produksi berubah → sinkronkan ke order + buat sale otomatis jika selesai
        static::updated(function ($production) {
            // Handle planning status approval - auto-set materials/spareparts status ke "preparing"
            if ($production->wasChanged('planning_status') && $production->planning_status === 'approved') {
                // Auto-set materials_status dan spareparts_status ke "preparing" jika ada materials/spareparts
                $hasMaterials = $production->productionMaterials()->count() > 0;
                $hasSpareparts = $production->productionSpareparts()->count() > 0;

                $updates = [];
                if ($hasMaterials && $production->materials_status === 'pending') {
                    $updates['materials_status'] = 'preparing';
                }
                if ($hasSpareparts && $production->spareparts_status === 'pending') {
                    $updates['spareparts_status'] = 'preparing';
                }

                if (!empty($updates)) {
                    $production->updateQuietly($updates);
                }
            }

            if ($production->wasChanged('completion_status') && $production->completion_status === 'approved') {
                $production->updateQuietly([
                    'status' => 'selesai',
                    'end_date' => $production->end_date ?? now(),
                ]);

                $production->order?->updateQuietly([
                    'production_status' => 'selesai'
                ]);

                $kdProject = $production->kd_project;
                $project = null;
                $namaProyek = null;

                if ($kdProject) {
                    $project = Project::where('kode_bantu', $kdProject)->first();
                }

                if ((!$project || !$kdProject) && $production->order && method_exists($production->order, 'ensureProjectExists')) {
                    $project = $production->order->ensureProjectExists();
                    if ($project && !$kdProject) {
                        $kdProject = $project->kode_bantu;
                    }
                }

                if ($project && !$kdProject) {
                    $kdProject = $project->kode_bantu;
                }

                if ($project) {
                    $namaProyek = $project->nama_proyek;
                }

                $totalLabor = $production->total_labor_cost;

                if ($totalLabor > 0) {
                    $laborCode = 'PRD-' . $production->id . '-LABOR';
                    $existsLaborJournal = JournalEntry::where('kode', $laborCode)->exists();

                    if (!$existsLaborJournal) {
                        $tanggalTransaksi = $production->completion_approved_at ?? now();

                        JournalEntry::create([
                            'tanggal_transaksi' => $tanggalTransaksi,
                            'bukti_transaksi' => null,
                            'keterangan' => 'Biaya tenaga kerja ' . ($namaProyek ?? ($production->product->product_title ?? '')),
                            'kd_perk' => '501-02',
                            'customer' => $production->order?->customer_name,
                            'kode' => $laborCode,
                            'nama_proyek' => $namaProyek,
                            'kd_project' => $kdProject,
                            'debit' => $totalLabor,
                            'kredit' => null,
                            'created_by' => $production->order?->user_id,
                        ]);

                        JournalEntry::create([
                            'tanggal_transaksi' => $tanggalTransaksi,
                            'bukti_transaksi' => null,
                            'keterangan' => 'Biaya tenaga kerja ' . ($namaProyek ?? ($production->product->product_title ?? '')),
                            'kd_perk' => '101-01',
                            'customer' => $production->order?->customer_name,
                            'kode' => $laborCode,
                            'nama_proyek' => $namaProyek,
                            'kd_project' => $kdProject,
                            'debit' => null,
                            'kredit' => $totalLabor,
                            'created_by' => $production->order?->user_id,
                        ]);
                    }
                }
            }

            if ($production->wasChanged('status')) {
                // Sinkronkan status produksi ke order.production_status
                // Status production tidak bergantung pada payment status, tapi payment status bisa mempengaruhi penjualan
                $production->order?->updateQuietly([
                    'production_status' => $production->status
                ]);

                // CATATAN: Sale TIDAK dibuat otomatis di sini
                // Sale hanya dibuat ketika customer mengkonfirmasi serah terima
                // melalui CustomerConfirmationController atau HandoverController forceComplete
            }
        });

        // Saat produksi dihapus → order kembali ke "menunggu"
        static::deleted(function ($production) {
            $production->order?->updateQuietly([
                'production_status' => 'menunggu'
            ]);
        });
    }


    /*
    |--------------------------------------------------------------------------
    | STATUS LABEL (untuk tampilan)
    |--------------------------------------------------------------------------
    */
    public function getStatusLabelAttribute()
    {
        return match ($this->status) {
            'menunggu'      => 'Menunggu Produksi',
            'dalam_proses'  => 'Sedang Diproduksi',
            'selesai'       => 'Produksi Selesai',
            'dibatalkan'    => 'Produksi Dibatalkan',
            default         => ucfirst($this->status),
        };
    }

    public function getPlanningStatusLabelAttribute()
    {
        return match ($this->planning_status) {
            'planning'          => 'Sedang Planning',
            'pending_approval'  => 'Menunggu Approval',
            'approved'          => 'Planning Disetujui',
            'rejected'          => 'Planning Ditolak',
            default             => ucfirst($this->planning_status),
        };
    }

    /*
    |--------------------------------------------------------------------------
    | SCOPES
    |--------------------------------------------------------------------------
    */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Custom route model binding untuk teknisi
     * Memastikan hanya produksi yang ditugaskan ke teknisi yang bisa diakses
     */
    public function resolveRouteBinding($value, $field = null)
    {
        $production = parent::resolveRouteBinding($value, $field);

        if (!$production) {
            return null;
        }

        // Cek apakah route ini adalah route teknisi
        try {
            $routeName = request()->route()->getName();
            $isTeknisiRoute = str_starts_with($routeName, 'teknisi.productions.');

            if ($isTeknisiRoute && \Illuminate\Support\Facades\Auth::check()) {
                $user = \Illuminate\Support\Facades\Auth::user();

                // Pastikan user adalah teknisi
                if ($user->user_type !== 'teknisi') {
                    return null;
                }

                // Pastikan produksi ditugaskan ke teknisi yang login
                if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)$user->id) {
                    return null; // Return null akan trigger 404, bukan 403
                }
            }
        } catch (\Exception $e) {
            // Jika ada error saat cek route, skip authorization check
            // Log error untuk debugging
            \Illuminate\Support\Facades\Log::warning('Error in Production route model binding: ' . $e->getMessage());
        }

        return $production;
    }

    /*
    |--------------------------------------------------------------------------
    | ESTIMATION HELPERS
    |--------------------------------------------------------------------------
    */
    /**
     * Get total estimated days (preparation + duration)
     */
    public function getTotalEstimatedDaysAttribute()
    {
        $prep = $this->material_preparation_days ?? 2;
        $duration = $this->estimated_duration_days ?? 0;
        return $prep + $duration;
    }

    /**
     * Calculate estimated completion date based on actual start
     */
    public function calculateEstimatedCompletionDate()
    {
        if ($this->actual_start_date && $this->estimated_duration_days) {
            $this->updateQuietly([
                'estimated_completion_date' => $this->actual_start_date->addDays($this->estimated_duration_days)
            ]);
        }
    }

    /**
     * Get materials status label
     */
    public function getMaterialsStatusLabelAttribute()
    {
        return match ($this->materials_status) {
            'pending' => 'Menunggu Persiapan',
            'preparing' => 'Sedang Disiapkan',
            'received' => 'Sudah Diterima',
            'rejected' => 'Ditolak',
            default => ucfirst($this->materials_status ?? 'Pending'),
        };
    }

    /**
     * Convert duration string to integer days
     * Examples: "2 minggu" -> 14, "7 hari" -> 7, "1 bulan" -> 30, etc.
     */
    public static function parseDurationToDays(?string $duration): int
    {
        if (!$duration) {
            return 7; // Default 7 hari
        }

        // Remove extra spaces and convert to lowercase
        $duration = strtolower(trim($duration));

        // Extract number and unit
        preg_match('/(\d+)\s*(minggu|hari|bulan|week|day|month)?/i', $duration, $matches);

        if (empty($matches) || !isset($matches[1])) {
            // If no match, try to extract just the number
            preg_match('/(\d+)/', $duration, $numMatches);
            if (isset($numMatches[1])) {
                return (int) $numMatches[1]; // Assume it's already in days
            }
            return 7; // Default
        }

        $number = (int) $matches[1];
        $unit = isset($matches[2]) ? strtolower($matches[2]) : 'hari';

        // Convert to days based on unit
        return match ($unit) {
            'minggu', 'week' => $number * 7,
            'bulan', 'month' => $number * 30,
            'hari', 'day', '' => $number,
            default => $number, // Default assume days
        };
    }

    /**
     * Get spareparts status label
     */
    public function getSparepartsStatusLabelAttribute()
    {
        return match ($this->spareparts_status) {
            'pending' => 'Menunggu Persiapan',
            'preparing' => 'Sedang Disiapkan',
            'received' => 'Sudah Diterima',
            'rejected' => 'Ditolak',
            default => ucfirst($this->spareparts_status ?? 'Pending'),
        };
    }
}
