<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Product extends Model
{
    use HasFactory;

    // Kolom yang boleh diisi (mass assignable)
    protected $fillable = [
        'product_title',
        'product_image',
        'product_description',
        'product_prices',
        'category_id',
        'product_work_duration',
        'product_garansi',
        'service_fee', // upah jasa
        'is_active', // default true
    ];

    protected $casts = [
        'product_prices' => 'integer',
        'service_fee' => 'integer',
        'is_active' => 'boolean',
    ];


    /**
     * Relasi ke kategori (opsional)
     * Jika product_category berisi id kategori dari tabel categories.
     */
    public function category()
    {
        return $this->belongsTo(Category::class, 'category_id');
    }

    public function getCategoryNameAttribute()
    {
        return $this->category ? ucfirst($this->category->name) : 'Umum';
    }

    /**
     * Format harga ke format Rupiah otomatis.
     */
    public function getFormattedPriceAttribute()
    {
        return 'Rp ' . number_format($this->product_prices, 0, ',', '.');
    }

    /**
     * Path lengkap gambar produk (jika disimpan di storage/public/products)
     */
    public function getImageUrlAttribute()
    {
        if ($this->product_image) {
            return asset('storage/products/' . $this->product_image);
        }

        // Jika tidak ada, ambil dari relasi images
        $firstImage = $this->images()->first();
        if ($firstImage) {
            return asset('storage/products/' . $firstImage->image_path);
        }

        return asset('images/no-image.png'); // gambar default
    }

    // Relasi ke tabel specifications
    public function specifications()
    {
        return $this->hasMany(ProductSpecification::class);
    }

    // Relasi ke tabel images
    public function images()
    {
        return $this->hasMany(ProductImage::class);
    }

    /**
     * Relasi ke Order
     * Satu produk bisa dibeli di banyak pesanan
     */
    public function orders()
    {
        return $this->hasMany(Order::class, 'product_id');
    }

    // relasi ke materials (sekarang dari tabel pivot langsung, bukan foreign key)
    public function materials()
    {
        // Menggunakan query builder untuk mengambil data dari tabel pivot
        return $this->hasMany(\App\Models\ProductMaterial::class, 'product_id');
    }

    // relasi ke spareparts (sekarang dari tabel pivot langsung, bukan foreign key)
    public function spareparts()
    {
        // Menggunakan query builder untuk mengambil data dari tabel pivot
        return $this->hasMany(\App\Models\ProductSparepart::class, 'product_id');
    }

    // relasi ke productions
    public function productions()
    {
        return $this->hasMany(Production::class);
    }

    // relasi ke penjualan
    public function sales()
    {
        return $this->hasMany(Sale::class);
    }

    /**
     * Check apakah produk masih tersedia
     */
    public function isAvailable()
    {
        // Semua produk selalu bisa dipesan, karena dibuat saat order.
        return true;
    }

    /**
     * Check apakah service ini perlu produksi
     * Semua service adalah make-to-order, jadi selalu perlu produksi
     */
    public function needsProduction()
    {
        return true;
    }
}
