<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class JournalEntry extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tanggal_transaksi',
        'bukti_transaksi',
        'keterangan',
        'nama_perkiraan',
        'kd_perk',
        'customer',
        'kode',
        'nama_proyek',
        'kd_project',
        'debit',
        'kredit',
        'created_by',
    ];

    protected $casts = [
        'tanggal_transaksi' => 'date',
        'debit' => 'decimal:2',
        'kredit' => 'decimal:2',
    ];

    /**
     * Relationship with User (creator)
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Relationship with ChartOfAccount
     */
    public function chartOfAccount(): BelongsTo
    {
        return $this->belongsTo(ChartOfAccount::class, 'kd_perk', 'kode_akun');
    }

    /**
     * Relationship with Project
     */
    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class, 'kd_project', 'kode_bantu');
    }

    /**
     * Scope for filtering by date range
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('tanggal_transaksi', [$startDate, $endDate]);
    }

    /**
     * Scope for filtering by project
     */
    public function scopeByProject($query, $projectCode)
    {
        return $query->where('kd_project', $projectCode);
    }

    /**
     * Scope for filtering by account code
     */
    public function scopeByAccount($query, $accountCode)
    {
        return $query->where('kd_perk', $accountCode);
    }

    /**
     * Get total debit for the entry
     */
    public function getTotalDebitAttribute()
    {
        return $this->debit ?? 0;
    }

    /**
     * Get total kredit for the entry
     */
    public function getTotalKreditAttribute()
    {
        return $this->kredit ?? 0;
    }

    /**
     * Check if entry is balanced (debit = kredit)
     */
    public function isBalanced(): bool
    {
        return $this->debit == $this->kredit;
    }
}
