<?php

namespace App\Http\Controllers\Supervisor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ProductInquiry;
use Illuminate\Support\Facades\Mail;

class QuotationController extends Controller
{
    public function index(Request $request)
    {
        $query = ProductInquiry::with(['product', 'supervisor'])
            ->where('status', 'supervisor_review')
            ->latest();
        
        // Cek apakah ada inquiry yang di-highlight
        $highlightedInquiry = null;
        $statusMessage = null;
        
        if ($request->has('highlight')) {
            $highlightId = $request->get('highlight');
            $highlightedInquiry = ProductInquiry::where('id', $highlightId)
                ->where('status', 'supervisor_review')
                ->first();
            
            // Jika inquiry tidak ditemukan atau status sudah berubah, tampilkan pesan
            if (!$highlightedInquiry) {
                $inquiry = ProductInquiry::find($highlightId);
                if ($inquiry) {
                    // Inquiry ada tapi status sudah berubah
                    $statusMessage = 'Penawaran #' . $highlightId . ' sudah diproses atau statusnya sudah berubah.';
                } else {
                    // Inquiry tidak ditemukan
                    $statusMessage = 'Penawaran #' . $highlightId . ' tidak ditemukan.';
                }
            }
        }
        
        $inquiries = $query->paginate(20);
        
        return view('supervisor.quotations.index', compact('inquiries', 'highlightedInquiry'))
            ->with('statusMessage', $statusMessage);
    }

    public function approve(Request $request, $id)
    {
        $request->validate([
            'min_price' => 'required|numeric|min:0',
            'max_price' => 'required|numeric|min:0|gte:min_price',
            'target_price' => 'nullable|numeric|min:0',
            'validity_until' => 'nullable|date|after:today',
            'supervisor_note' => 'nullable|string|max:1000',
        ]);

        $inquiry = ProductInquiry::findOrFail($id);

        $inquiry->update([
            'supervisor_floor_price' => $request->min_price,
            'supervisor_target_price' => $request->target_price ?? null,
            'supervisor_ceiling_price' => $request->max_price,
            'validity_until' => $request->validity_until,
            'min_price' => $request->min_price, // legacy fields kept
            'max_price' => $request->max_price,
            'supervisor_note' => $request->supervisor_note,
            'supervisor_id' => auth()->id(),
            'supervisor_approved_at' => now(),
            'status' => 'sent_to_admin',
            'is_read' => false, // Set is_read = false agar admin mendapat notifikasi
        ]);

        // Notifikasi admin via email
        try {
            Mail::send('emails.supervisor-range-set', [ 'inquiry' => $inquiry ], function ($mail) use ($inquiry) {
                $mail->to(config('mail.from.address'), 'Admin')
                    ->subject('Koridor Harga Diset - ' . ($inquiry->product->product_title ?? 'Produk'))
                    ->from(config('mail.from.address'), config('mail.from.name'));
            });
        } catch (\Exception $e) { }

        // Kirim notifikasi ke admin
        try {
            $notificationService = new \App\Services\NotificationService();
            $notificationService->notifySupervisorPriceCorridorSet($inquiry);
        } catch (\Exception $e) {
            \Log::error('Failed to send supervisor price corridor notification: ' . $e->getMessage());
        }

        return back()->with('success', 'Kebijakan harga diset. Menunggu admin mengirim final offer ke customer.');
    }

    public function reject(Request $request, $id)
    {
        $request->validate([
            'supervisor_note' => 'required|string|max:1000',
        ]);

        $inquiry = ProductInquiry::findOrFail($id);
        
        $inquiry->update([
            'supervisor_note' => $request->supervisor_note,
            'supervisor_id' => auth()->id(),
            'status' => 'rejected',
        ]);

        return back()->with('success', 'Penawaran telah ditolak.');
    }
}





