<?php

namespace App\Http\Controllers\Supervisor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Production;
use App\Models\ProductInquiry;
use Illuminate\Support\Facades\Auth;

class ProductionController extends Controller
{
    public function index()
    {
        // Redirect to planning by default
        return redirect()->route('supervisor.productions.planning');
    }

    public function planning()
    {
        // Planning pending approval
        $pendingPlanning = Production::with(['order', 'product', 'teknisi'])
            ->where('planning_status', 'pending_approval')
            ->latest('submitted_at')
            ->get();

        // Planning pending lebih dari 24 jam (alert)
        $pendingPlanningOverdue = Production::with(['order', 'product', 'teknisi'])
            ->where('planning_status', 'pending_approval')
            ->where('submitted_at', '<', now()->subDay()) // Lebih dari 24 jam
            ->latest('submitted_at')
            ->get();

        // Produksi yang melewati estimasi (dalam proses dan melewati estimated_completion_date)
        $overdueProductions = Production::with(['order', 'product', 'teknisi'])
            ->where('status', 'dalam_proses')
            ->whereNotNull('estimated_completion_date')
            ->where('estimated_completion_date', '<', now())
            ->latest('estimated_completion_date')
            ->get();
        
        return view('supervisor.productions.planning.index', compact(
            'pendingPlanning', 
            'pendingPlanningOverdue',
            'overdueProductions'
        ));
    }

    public function completion()
    {
        // Completion pending approval
        $pendingCompletion = Production::with(['order', 'product', 'teknisi'])
            ->where('completion_status', 'pending_approval')
            ->latest('completed_at')
            ->get();
        
        return view('supervisor.productions.completion.index', compact('pendingCompletion'));
    }

    public function show(Production $production)
    {
        $production->load([
            'order', 
            'product', 
            'teknisi', 
            'productionMaterials.material', 
            'productionSpareparts.sparepart',
            'itemRequests.material',
            'itemRequests.sparepart',
            'itemRequests.requester',
            'itemRequests.processor',
            'itemRequests.purchase'
        ]);

        // Gabungkan materials dan spareparts untuk unified table
        $allItems = collect();
        foreach ($production->productionMaterials as $pm) {
            $allItems->push([
                'type' => 'material',
                'id' => $pm->id,
                'name' => $pm->material->name ?? 'N/A',
                'quantity' => $pm->quantity,
                'unit' => $pm->unit,
                'unit_cost' => $pm->unit_cost,
                'total_cost' => $pm->total_cost,
                'stock' => $pm->material->stock ?? 0,
                'stock_status' => $pm->stock_status ?? 'tersedia',
                'is_received' => $pm->is_received ?? false,
                'is_additional' => $pm->is_additional ?? false,
            ]);
        }
        foreach ($production->productionSpareparts as $ps) {
            $allItems->push([
                'type' => 'sparepart',
                'id' => $ps->id,
                'name' => $ps->sparepart->name ?? 'N/A',
                'quantity' => $ps->quantity,
                'unit' => $ps->unit,
                'unit_cost' => $ps->unit_cost,
                'total_cost' => $ps->total_cost,
                'stock' => $ps->sparepart->stock ?? 0,
                'stock_status' => $ps->stock_status ?? 'tersedia',
                'is_received' => $ps->is_received ?? false,
                'is_additional' => $ps->is_additional ?? false,
            ]);
        }

        return view('supervisor.productions.show', compact('production', 'allItems'));
    }

    public function approve(Request $request, Production $production)
    {
        $request->validate([
            'feedback' => 'nullable|string|max:1000',
        ]);

        $production->update([
            'planning_status' => 'approved',
            'approved_at' => now(),
            'approved_by' => Auth::id(),
            'supervisor_feedback' => $request->feedback,
        ]);

        // Kirim notifikasi ke teknisi
        try {
            $notificationService = new \App\Services\NotificationService();
            $notificationService->notifyPlanningApproved($production->fresh());
        } catch (\Exception $e) {
            \Log::error('Failed to send planning approved notification: ' . $e->getMessage());
        }

        return back()->with('success', 'Planning produksi berhasil disetujui. Teknisi dapat mulai produksi.');
    }

    public function reject(Request $request, Production $production)
    {
        $request->validate([
            'feedback' => 'required|string|max:1000',
        ]);

        $production->update([
            'planning_status' => 'rejected',
            'supervisor_feedback' => $request->feedback,
        ]);

        // Kirim notifikasi ke teknisi
        try {
            $notificationService = new \App\Services\NotificationService();
            $notificationService->notifyPlanningRejected($production->fresh());
        } catch (\Exception $e) {
            \Log::error('Failed to send planning rejected notification: ' . $e->getMessage());
        }

        return back()->with('success', 'Planning produksi ditolak. Teknisi dapat merevisi planning.');
    }

    /**
     * Approve production completion
     * 
     * Supervisor menyetujui penyelesaian produksi yang diajukan teknisi.
     * Setelah disetujui, model event akan:
     * 1. Mengubah status produksi menjadi 'selesai'
     * 2. Mengupdate order.production_status menjadi 'selesai'
     * 
     * CATATAN: Sale TIDAK dibuat otomatis di sini.
     * Sale hanya dibuat setelah customer mengkonfirmasi serah terima melalui link konfirmasi
     * yang dikirim admin dari menu "Serah Terima".
     */
    public function approveCompletion(Request $request, Production $production)
    {
        // Pastikan completion status adalah pending_approval
        if ($production->completion_status !== 'pending_approval') {
            return back()->withErrors(['status' => 'Production completion ini tidak dalam status pending approval.']);
        }

        $request->validate([
            'feedback' => 'nullable|string|max:1000',
        ]);

        // Approval completion - status akan otomatis berubah via model event
        $production->update([
            'completion_status' => 'approved',
            'completion_approved_at' => now(),
            'completion_approved_by' => Auth::id(),
            'supervisor_feedback' => $request->feedback ?? $production->supervisor_feedback,
        ]);

        // Reload production dan order untuk mendapatkan data terbaru setelah event
        $production->refresh();
        $production->load('order');

        // Catatan: Berikut ini terjadi otomatis via model booted event:
        // 1. Status produksi berubah menjadi 'selesai'
        // 2. Order.production_status diupdate menjadi 'selesai'
        // Sale TIDAK dibuat otomatis - hanya dibuat setelah customer konfirmasi serah terima

        // Kirim notifikasi untuk completion approved
        try {
            $notificationService = new \App\Services\NotificationService();
            $notificationService->notifyCompletionApproved($production);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to send completion approved notification: ' . $e->getMessage());
        }

        $message = 'Production completion berhasil disetujui. Status produksi sekarang: Selesai.';
        $message .= ' Admin dapat mengirim link konfirmasi serah terima ke customer melalui menu "Serah Terima".';
        
        // Catatan: Sale akan dibuat setelah customer mengkonfirmasi serah terima
        \Log::info('Production completion approved - waiting for customer handover confirmation', [
                'production_id' => $production->id,
                'order_id' => $production->order->id,
                'order_status' => $production->order->status,
            'production_status' => $production->status,
            ]);

        return back()->with('success', $message);
    }

    /**
     * Reject production completion
     */
    public function rejectCompletion(Request $request, Production $production)
    {
        // Pastikan completion status adalah pending_approval
        if ($production->completion_status !== 'pending_approval') {
            return back()->withErrors(['status' => 'Production completion ini tidak dalam status pending approval.']);
        }

        $request->validate([
            'feedback' => 'required|string|max:1000',
        ]);

        $production->update([
            'completion_status' => 'rejected',
            'status' => 'dalam_proses', // Kembalikan ke dalam proses
            'end_date' => null,
            'completed_at' => null,
            'supervisor_feedback' => $request->feedback,
        ]);

        // Kirim notifikasi ke teknisi
        try {
            $notificationService = new \App\Services\NotificationService();
            $notificationService->notifyCompletionRejected($production->fresh());
        } catch (\Exception $e) {
            \Log::error('Failed to send completion rejected notification: ' . $e->getMessage());
        }

        return back()->with('success', 'Production completion ditolak. Teknisi dapat memperbaiki dan submit ulang.');
    }
}