<?php

namespace App\Http\Controllers\Supervisor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Production;
use App\Models\ProductInquiry;
use App\Models\Order;
use Carbon\Carbon;

class CalendarController extends Controller
{
    public function index(Request $request)
    {
        $startDate = $request->get('start', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $type = $request->get('type', 'all'); // all, production, quotation, order

        // Events untuk Produksi
        $productionEvents = [];
        if ($type === 'all' || $type === 'production') {
            $productions = Production::with(['order', 'product', 'teknisi'])
                ->where(function($query) use ($startDate, $endDate) {
                    $query->whereBetween('estimated_completion_date', [$startDate, $endDate])
                          ->orWhereBetween('start_date', [$startDate, $endDate])
                          ->orWhereBetween('created_at', [$startDate, $endDate]);
                })
                ->get();

            foreach ($productions as $production) {
                // Deadline completion
                if ($production->estimated_completion_date) {
                    $productionEvents[] = [
                        'id' => 'prod_' . $production->id . '_deadline',
                        'title' => 'Deadline: ' . ($production->product->product_title ?? 'Produk #' . $production->id),
                        'start' => $production->estimated_completion_date->format('Y-m-d'),
                        'end' => $production->estimated_completion_date->format('Y-m-d'),
                        'type' => 'production',
                        'color' => $production->status === 'selesai' ? '#10b981' : ($production->status === 'dalam_proses' ? '#3b82f6' : '#f59e0b'),
                        'url' => route('supervisor.productions.show', $production->id),
                        'description' => 'Produksi #' . $production->id . ' - ' . ($production->order->customer_name ?? 'N/A'),
                        'status' => $production->status,
                    ];
                }

                // Start date
                if ($production->start_date) {
                    $productionEvents[] = [
                        'id' => 'prod_' . $production->id . '_start',
                        'title' => 'Mulai: ' . ($production->product->product_title ?? 'Produk #' . $production->id),
                        'start' => $production->start_date->format('Y-m-d'),
                        'end' => $production->start_date->format('Y-m-d'),
                        'type' => 'production',
                        'color' => '#8b5cf6',
                        'url' => route('supervisor.productions.show', $production->id),
                        'description' => 'Produksi #' . $production->id,
                        'status' => $production->status,
                    ];
                }

                // Planning pending (lebih dari 24 jam)
                if ($production->planning_status === 'pending_approval' && $production->submitted_at) {
                    $daysPending = $production->submitted_at->diffInDays(now());
                    if ($daysPending >= 1) {
                        $productionEvents[] = [
                            'id' => 'prod_' . $production->id . '_pending',
                            'title' => '⚠ Pending Approval: ' . ($production->product->product_title ?? 'Produk #' . $production->id),
                            'start' => $production->submitted_at->format('Y-m-d'),
                            'end' => $production->submitted_at->format('Y-m-d'),
                            'type' => 'production',
                            'color' => '#ef4444',
                            'url' => route('supervisor.productions.planning'),
                            'description' => 'Planning pending ' . $daysPending . ' hari',
                            'status' => 'pending',
                        ];
                    }
                }
            }
        }

        // Events untuk Penawaran (Quotation)
        $quotationEvents = [];
        if ($type === 'all' || $type === 'quotation') {
            $quotations = ProductInquiry::with(['product'])
                ->where('status', 'supervisor_review')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->get();

            foreach ($quotations as $quotation) {
                // Deadline review (asumsi 3 hari dari created_at)
                $reviewDeadline = $quotation->created_at->copy()->addDays(3);
                $daysRemaining = now()->diffInDays($reviewDeadline, false);
                
                $quotationEvents[] = [
                    'id' => 'quote_' . $quotation->id,
                    'title' => 'Review: ' . ($quotation->product->product_title ?? 'Penawaran #' . $quotation->id),
                    'start' => $reviewDeadline->format('Y-m-d'),
                    'end' => $reviewDeadline->format('Y-m-d'),
                    'type' => 'quotation',
                    'color' => $daysRemaining < 0 ? '#ef4444' : ($daysRemaining <= 1 ? '#f59e0b' : '#3b82f6'),
                    'url' => route('supervisor.quotations.index', ['highlight' => $quotation->id]),
                    'description' => 'Penawaran #' . $quotation->id . ' - ' . ($quotation->customer_name ?? 'N/A'),
                    'status' => $quotation->status,
                    'days_remaining' => $daysRemaining,
                ];
            }
        }

        // Events untuk Order
        $orderEvents = [];
        if ($type === 'all' || $type === 'order') {
            $orders = Order::with(['product', 'user'])
                ->whereBetween('created_at', [$startDate, $endDate])
                ->get();

            foreach ($orders as $order) {
                if ($order->order_status === 'menunggu' || $order->order_status === 'diterima') {
                    $orderEvents[] = [
                        'id' => 'order_' . $order->id,
                        'title' => 'Order: ' . ($order->product->product_title ?? 'Order #' . $order->id),
                        'start' => $order->created_at->format('Y-m-d'),
                        'end' => $order->created_at->format('Y-m-d'),
                        'type' => 'order',
                        'color' => $order->order_status === 'menunggu' ? '#f59e0b' : '#10b981',
                        'url' => route('supervisor.monitoring.orders'),
                        'description' => 'Order #' . $order->id . ' - ' . ($order->user->name ?? 'N/A'),
                        'status' => $order->order_status,
                    ];
                }
            }
        }

        // Gabungkan semua events
        $events = array_merge($productionEvents, $quotationEvents, $orderEvents);

        // Statistik
        $stats = [
            'total_events' => count($events),
            'production_events' => count($productionEvents),
            'quotation_events' => count($quotationEvents),
            'order_events' => count($orderEvents),
            'overdue_count' => collect($events)->filter(function($event) {
                return isset($event['days_remaining']) && $event['days_remaining'] < 0;
            })->count(),
        ];

        return view('supervisor.calendar.index', compact('events', 'stats', 'startDate', 'endDate', 'type'));
    }
}

