<?php

namespace App\Http\Controllers\Supervisor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\Production;
use App\Models\Purchase;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ApprovalController extends Controller
{
    public function index()
    {
        // Statistik untuk dashboard summary
        // CATATAN: Order approval dilakukan oleh Admin, bukan Supervisor
        // Supervisor hanya approve Production Planning dan Purchase
        $stats = [
            // Pending Productions (Planning & Completion)
            'pending_planning_count' => Production::where('planning_status', 'pending_approval')->count(),
            'pending_completion_count' => Production::where('completion_status', 'pending_approval')->count(),
            'total_productions' => Production::where(function($query) {
                $query->where('planning_status', 'pending_approval')
                      ->orWhere('completion_status', 'pending_approval');
            })->count(),
            
            // Pending Purchases
            'pending_purchases_count' => Purchase::where('status', 'pending')->count(),
        ];

        return view('supervisor.approvals.index', compact('stats'));
    }

    public function orders()
    {
        // Hanya tampilkan orders dengan status 'menunggu' yang perlu di-approve
        $orders = Order::with(['product', 'user'])
            ->where('order_status', 'menunggu')
            ->latest()
            ->paginate(20);
        
        return view('supervisor.approvals.orders', compact('orders'));
    }

    public function approveOrder(Request $request, $order)
    {
        $request->validate([
            'feedback' => 'nullable|string|max:1000',
        ]);

        $order = Order::findOrFail($order);
        
        // Update order status
        $order->update([
            'order_status' => 'diterima'
        ]);

        // Buat produksi jika order diterima dan payment verified (menggunakan method terpusat)
        // Note: Jika payment belum verified, production tidak akan dibuat
        // Production akan dibuat otomatis saat payment verified jika order sudah diterima
        $order->ensureProductionExists();

        return back()->with('success', 'Order #' . $order->id . ' berhasil diapprove. Production akan dibuat otomatis.');
    }

    public function approveProduction($production)
    {
        // Logic ini akan di-handle di ProductionController
        return back()->with('success', 'Production berhasil diapprove');
    }

    public function approvePurchase(Request $request, $purchase)
    {
        $request->validate([
            'feedback' => 'nullable|string|max:1000',
        ]);

        $purchase = Purchase::with('items')->findOrFail($purchase);
        
        DB::beginTransaction();
        try {
            // Update status purchase
            $purchase->update([
                'status' => 'approved',
                'approved_at' => now(),
                'approved_by' => Auth::id(),
                'supervisor_feedback' => $request->feedback,
            ]);

            // Update stock untuk setiap item di purchase
            foreach ($purchase->items as $purchaseItem) {
                $quantity = $purchaseItem->quantity;
                $itemId = $purchaseItem->item_id;
                
                // Gunakan item_type dari purchaseItem untuk menentukan material atau sparepart
                if ($purchaseItem->item_type === 'material') {
                    $material = \App\Models\Material::find($itemId);
                    if ($material) {
                        $material->increment('stock', $quantity);
                    }
                } elseif ($purchaseItem->item_type === 'sparepart') {
                    $sparepart = \App\Models\Sparepart::find($itemId);
                    if ($sparepart) {
                        $sparepart->increment('stock', $quantity);
                    }
                }
            }

            DB::commit();
            $kdProject = null;
            $namaProyek = null;
            $linkedIr = \App\Models\ProductionItemRequest::where('purchase_id', $purchase->id)->first();
            if ($linkedIr && $linkedIr->production) {
                $kdProject = $linkedIr->production->kd_project;
                $project = \App\Models\Project::where('kode_bantu', $kdProject)->first();
                $namaProyek = $project ? $project->nama_proyek : null;
            }

            $purchaseCode = 'PUR-' . $purchase->id;
            $existsCredit = \App\Models\JournalEntry::where('kode', $purchaseCode)->where('kd_perk', '101-01')->exists();
            if (!$existsCredit) {
                foreach ($purchase->items as $pi) {
                    $itemCode = $purchaseCode . '-ITEM-' . $pi->id;
                    $existsItem = \App\Models\JournalEntry::where('kode', $itemCode)->exists();
                    if ($existsItem) { continue; }
                    \App\Models\JournalEntry::create([
                        'tanggal_transaksi' => $purchase->purchase_date ?? now(),
                        'bukti_transaksi' => $purchase->purchase_code,
                        'keterangan' => 'Pembelian ' . ($pi->item_name ?? '-') . ' ' . ($pi->quantity ?? 0) . ' ' . ($pi->unit ?? '') . ' untuk ' . ($namaProyek ?? ''),
                        'kd_perk' => '501-01',
                        'customer' => $purchase->supplier_name,
                        'kode' => $itemCode,
                        'nama_proyek' => $namaProyek,
                        'kd_project' => $kdProject,
                        'debit' => $pi->total_price,
                        'kredit' => null,
                        'created_by' => Auth::id(),
                    ]);
                }
                \App\Models\JournalEntry::create([
                    'tanggal_transaksi' => $purchase->purchase_date ?? now(),
                    'bukti_transaksi' => $purchase->purchase_code,
                    'keterangan' => 'Pembelian bahan/sparepart',
                    'kd_perk' => '101-01',
                    'customer' => $purchase->supplier_name,
                    'kode' => $purchaseCode,
                    'nama_proyek' => $namaProyek,
                    'kd_project' => $kdProject,
                    'debit' => null,
                    'kredit' => $purchase->total_amount,
                    'created_by' => Auth::id(),
                ]);
            }

            return back()->with('success', 'Purchase #' . $purchase->id . ' berhasil diapprove, stock diperbarui, dan jurnal otomatis dibuat.');
        } catch (\Exception $e) {
            DB::rollBack();
            \Illuminate\Support\Facades\Log::error('Purchase Approval Error: ' . $e->getMessage());
            return back()->with('error', 'Gagal menyetujui purchase: ' . $e->getMessage());
        }
    }

    public function rejectOrder(Request $request, $order)
    {
        $request->validate([
            'feedback' => 'required|string|max:1000',
        ]);

        $order = Order::findOrFail($order);
        
        $order->update([
            'order_status' => 'ditolak'
        ]);

        return back()->with('success', 'Order #' . $order->id . ' ditolak.');
    }

    public function rejectPurchase(Request $request, $purchase)
    {
        $request->validate([
            'feedback' => 'required|string|max:1000',
        ]);

        $purchase = Purchase::findOrFail($purchase);
        
        $purchase->update([
            'status' => 'rejected',
            'supervisor_feedback' => $request->feedback,
        ]);

        return back()->with('success', 'Purchase #' . $purchase->id . ' ditolak.');
    }
}






