<?php

namespace App\Http\Controllers\Supervisor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Production;
use App\Models\ProductInquiry;
use App\Models\Order;
use App\Models\Sale;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AnalyticsController extends Controller
{
    public function index(Request $request)
    {
        try {
            $period = $request->get('period', 'month'); // month, quarter, year
            $startDate = $request->get('start_date');
            $endDate = $request->get('end_date');

            // Set default date range berdasarkan period
            if (!$startDate || !$endDate) {
                switch ($period) {
                    case 'quarter':
                        $startDate = Carbon::now()->startOfQuarter()->format('Y-m-d');
                        $endDate = Carbon::now()->endOfQuarter()->format('Y-m-d');
                        break;
                    case 'year':
                        $startDate = Carbon::now()->startOfYear()->format('Y-m-d');
                        $endDate = Carbon::now()->endOfYear()->format('Y-m-d');
                        break;
                    default: // month
                        $startDate = Carbon::now()->startOfMonth()->format('Y-m-d');
                        $endDate = Carbon::now()->endOfMonth()->format('Y-m-d');
                }
            }

        // ========== PRODUCTION ANALYTICS ==========
        
        // Total Produksi
        $totalProductions = 0;
        try {
            $totalProductions = Production::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])->count();
        } catch (\Exception $e) {
            \Log::warning('Error counting total productions in analytics: ' . $e->getMessage());
        }
        
        // Produksi per Status
        $productionsByStatus = collect();
        try {
            $productionsByStatus = Production::select('status', DB::raw('count(*) as count'))
                ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
                ->groupBy('status')
                ->get()
                ->pluck('count', 'status');
        } catch (\Exception $e) {
            \Log::warning('Error getting productions by status in analytics: ' . $e->getMessage());
        }

        // Produksi per Bulan (untuk chart)
        $productionsByMonth = collect();
        try {
            $productionsByMonth = Production::select(
                    DB::raw('MONTH(created_at) as month'),
                    DB::raw('YEAR(created_at) as year'),
                    DB::raw('COUNT(*) as count')
                )
                ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
                ->groupBy('year', 'month')
                ->orderBy('year', 'asc')
                ->orderBy('month', 'asc')
                ->get();
        } catch (\Exception $e) {
            \Log::warning('Error getting productions by month in analytics: ' . $e->getMessage());
        }

        // Average Production Time (dalam hari)
        $avgProductionDays = 0;
        try {
            $avgProductionTime = Production::select(
                    DB::raw('AVG(DATEDIFF(COALESCE(end_date, NOW()), start_date)) as avg_days')
                )
                ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
                ->whereNotNull('start_date')
                ->where('status', 'selesai')
                ->first();
            
            $avgProductionDays = $avgProductionTime && $avgProductionTime->avg_days ? $avgProductionTime->avg_days : 0;
        } catch (\Exception $e) {
            \Log::warning('Error calculating average production time in analytics: ' . $e->getMessage());
        }

        // Total Biaya Produksi
        $totalProductionCost = (object)['total' => 0];
        try {
            $totalProductionCost = Production::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
                ->selectRaw('SUM(COALESCE(total_production_cost, 0) + COALESCE(labor_cost, 0)) as total')
                ->first();
            if (!$totalProductionCost) {
                $totalProductionCost = (object)['total' => 0];
            }
        } catch (\Exception $e) {
            \Log::warning('Error calculating total production cost in analytics: ' . $e->getMessage());
        }

        // ========== TEKNISI PERFORMANCE ==========
        
        $teknisiPerformance = collect();
        try {
            // Ambil semua teknisi
            $teknisiList = User::where('user_type', 'teknisi')->get();
            
            $teknisiPerformance = $teknisiList->map(function($teknisi) use ($startDate, $endDate) {
                // Hitung produksi secara manual karena tidak ada relasi
                $allProductions = Production::where('teknisi_id', $teknisi->id)
                    ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
                    ->get();
                
                $completedProductions = Production::where('teknisi_id', $teknisi->id)
                    ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
                    ->where('status', 'selesai')
                    ->get();
                
                $totalCost = $completedProductions->sum(function($prod) {
                    return ($prod->total_production_cost ?? 0) + ($prod->labor_cost ?? 0);
                });
                
                return [
                    'id' => $teknisi->id,
                    'name' => $teknisi->name,
                    'total_productions' => $allProductions->count(),
                    'completed_productions' => $completedProductions->count(),
                    'total_cost' => $totalCost,
                    'avg_cost_per_production' => $completedProductions->count() > 0 ? $totalCost / $completedProductions->count() : 0,
                ];
            })
            ->sortByDesc('completed_productions')
            ->take(10)
            ->values();
        } catch (\Exception $e) {
            \Log::error('Error loading teknisi performance in analytics: ' . $e->getMessage());
            $teknisiPerformance = collect();
        }

        // ========== QUOTATION ANALYTICS ==========
        
        $totalQuotations = 0;
        $pendingQuotations = 0;
        $approvedQuotations = 0;
        $rejectedQuotations = 0;
        try {
            $totalQuotations = ProductInquiry::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])->count();
            $pendingQuotations = ProductInquiry::where('status', 'supervisor_review')
                ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
                ->count();
            $approvedQuotations = ProductInquiry::where('status', 'approved')
                ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
                ->count();
            $rejectedQuotations = ProductInquiry::where('status', 'rejected')
                ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
                ->count();
        } catch (\Exception $e) {
            \Log::warning('Error counting quotations in analytics: ' . $e->getMessage());
        }

        // Average Review Time (dalam hari)
        $avgReviewDays = 0;
        try {
            $avgReviewTime = ProductInquiry::select(
                    DB::raw('AVG(DATEDIFF(COALESCE(updated_at, NOW()), created_at)) as avg_days')
                )
                ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
                ->whereIn('status', ['approved', 'rejected'])
                ->first();
            
            $avgReviewDays = $avgReviewTime && $avgReviewTime->avg_days ? $avgReviewTime->avg_days : 0;
        } catch (\Exception $e) {
            \Log::warning('Error calculating average review time in analytics: ' . $e->getMessage());
        }

        // ========== APPROVAL ANALYTICS ==========
        
        $pendingPlanning = 0;
        $pendingCompletion = 0;
        try {
            $pendingPlanning = Production::where('planning_status', 'pending_approval')
                ->whereBetween('submitted_at', [$startDate, $endDate . ' 23:59:59'])
                ->count();
            
            $pendingCompletion = Production::where('completion_status', 'pending_approval')
                ->whereBetween('submitted_at', [$startDate, $endDate . ' 23:59:59'])
                ->count();
        } catch (\Exception $e) {
            \Log::warning('Error counting pending approvals in analytics: ' . $e->getMessage());
        }

        // Average Approval Time (dalam jam)
        $avgApprovalHours = 0;
        try {
            $avgApprovalTime = Production::select(
                    DB::raw('AVG(TIMESTAMPDIFF(HOUR, submitted_at, updated_at)) as avg_hours')
                )
                ->whereBetween('submitted_at', [$startDate, $endDate . ' 23:59:59'])
                ->whereIn('planning_status', ['approved', 'rejected'])
                ->whereNotNull('submitted_at')
                ->first();
            
            $avgApprovalHours = $avgApprovalTime && $avgApprovalTime->avg_hours ? $avgApprovalTime->avg_hours : 0;
        } catch (\Exception $e) {
            \Log::warning('Error calculating average approval time in analytics: ' . $e->getMessage());
        }

        // ========== CHART DATA ==========
        
        // Production Trend (12 bulan terakhir)
        $productionTrend = collect();
        try {
            $productionTrend = Production::select(
                    DB::raw('MONTH(created_at) as month'),
                    DB::raw('YEAR(created_at) as year'),
                    DB::raw('COUNT(*) as count'),
                    DB::raw('SUM(COALESCE(total_production_cost, 0) + COALESCE(labor_cost, 0)) as total_cost')
                )
                ->where('created_at', '>=', Carbon::now()->subMonths(12))
                ->groupBy('year', 'month')
                ->orderBy('year', 'asc')
                ->orderBy('month', 'asc')
                ->get();
        } catch (\Exception $e) {
            \Log::warning('Error getting production trend in analytics: ' . $e->getMessage());
        }

        // Quotation Status Distribution
        $quotationStatusData = collect();
        try {
            $quotationStatusData = ProductInquiry::select('status', DB::raw('count(*) as count'))
                ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
                ->groupBy('status')
                ->get();
        } catch (\Exception $e) {
            \Log::warning('Error getting quotation status data in analytics: ' . $e->getMessage());
        }

        // ========== SUMMARY STATS ==========
        
        $summaryStats = [
            'total_productions' => $totalProductions,
            'completed_productions' => $productionsByStatus->get('selesai', 0),
            'in_progress_productions' => $productionsByStatus->get('dalam_proses', 0),
            'pending_productions' => $productionsByStatus->get('menunggu', 0),
            'total_production_cost' => $totalProductionCost->total ?? 0,
            'avg_production_time' => round($avgProductionDays, 1),
            'total_quotations' => $totalQuotations,
            'pending_quotations' => $pendingQuotations,
            'approved_quotations' => $approvedQuotations,
            'rejected_quotations' => $rejectedQuotations,
            'avg_review_time' => round($avgReviewDays, 1),
            'pending_planning' => $pendingPlanning,
            'pending_completion' => $pendingCompletion,
            'avg_approval_time' => round($avgApprovalHours, 1),
        ];

        return view('supervisor.analytics.index', compact(
            'summaryStats',
            'productionsByStatus',
            'productionsByMonth',
            'teknisiPerformance',
            'productionTrend',
            'quotationStatusData',
            'startDate',
            'endDate',
            'period'
        ));
        } catch (\Exception $e) {
            \Log::error('Analytics Controller Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            // Return view with empty data
            return view('supervisor.analytics.index', [
                'summaryStats' => [
                    'total_productions' => 0,
                    'completed_productions' => 0,
                    'in_progress_productions' => 0,
                    'pending_productions' => 0,
                    'total_production_cost' => 0,
                    'avg_production_time' => 0,
                    'total_quotations' => 0,
                    'pending_quotations' => 0,
                    'approved_quotations' => 0,
                    'rejected_quotations' => 0,
                    'avg_review_time' => 0,
                    'pending_planning' => 0,
                    'pending_completion' => 0,
                    'avg_approval_time' => 0,
                ],
                'productionsByStatus' => collect(),
                'productionsByMonth' => collect(),
                'teknisiPerformance' => collect(),
                'productionTrend' => collect(),
                'quotationStatusData' => collect(),
                'startDate' => Carbon::now()->startOfMonth()->format('Y-m-d'),
                'endDate' => Carbon::now()->endOfMonth()->format('Y-m-d'),
                'period' => 'month',
            ])->with('error', 'Terjadi kesalahan saat memuat data analytics: ' . $e->getMessage());
        }
    }
}

