<?php

namespace App\Http\Controllers;

use App\Http\Requests\ProfileUpdateRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Storage;
use Illuminate\View\View;

class ProfileController extends Controller
{
    /**
     * Display the user's profile form.
     */
    public function edit(Request $request): View
    {
        $user = $request->user();
        $userType = $user->user_type;
        
        // For regular users, use the dedicated user profile page
        if ($userType === 'user' || is_null($userType)) {
            return view('user.profile.edit', [
                'user' => $user,
            ]);
        }
        
        // For admin, supervisor, teknisi - use the original profile page with sidebar layout
        return view('profile.edit', [
            'user' => $user,
        ]);
    }

    /**
     * Update the user's profile information.
     */
    public function update(ProfileUpdateRequest $request): RedirectResponse
    {
        $user = $request->user();
        
        // Handle photo upload
        if ($request->hasFile('photo')) {
            try {
                // Delete old photo if exists
                if ($user->photo) {
                    $oldPhotoPath = public_path('uploads/profiles/' . $user->photo);
                    if (file_exists($oldPhotoPath)) {
                        @unlink($oldPhotoPath);
                    }
                }
                
                // Ensure directory exists
                $uploadPath = public_path('uploads/profiles');
                if (!is_dir($uploadPath)) {
                    mkdir($uploadPath, 0755, true);
                }
                
                // Store new photo
                $photo = $request->file('photo');
                $photoName = time() . '_' . uniqid() . '.' . $photo->getClientOriginalExtension();
                
                if ($photo->move($uploadPath, $photoName)) {
                    $user->photo = $photoName;
                } else {
                    return Redirect::back()->withErrors(['photo' => 'Gagal mengunggah foto. Pastikan folder uploads/profiles dapat ditulis.'])->withInput();
                }
            } catch (\Exception $e) {
                return Redirect::back()->withErrors(['photo' => 'Terjadi kesalahan saat mengunggah foto: ' . $e->getMessage()])->withInput();
            }
        }
        
        // Update other fields
        $user->name = $request->name;
        $user->email = $request->email;
        $user->phone = $request->phone;
        $user->address = $request->address;
        $user->postal_code = $request->postal_code;

        if ($user->isDirty('email')) {
            $user->email_verified_at = null;
        }

        $user->save();

        return Redirect::route('profile.edit')->with('status', 'profile-updated');
    }

    /**
     * Delete the user's account.
     */
    public function destroy(Request $request): RedirectResponse
    {
        $request->validateWithBag('userDeletion', [
            'password' => ['required', 'current_password'],
        ]);

        $user = $request->user();

        Auth::logout();

        $user->delete();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return Redirect::to('/');
    }
}
