<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Illuminate\View\View;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): View
    {
        return view('auth.register');
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        // Validasi input dengan rules yang lebih ketat
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255', 'regex:/^[a-zA-Z0-9\s\.\-\']+$/u'], // Hanya karakter alfanumerik dan beberapa karakter khusus
            'email' => ['required', 'string', 'lowercase', 'email:rfc,dns', 'max:255', 'unique:' . User::class], // Validasi email dengan DNS check
            'password' => [
                'required',
                'confirmed',
                Rules\Password::min(8)
                    ->letters() // Minimal harus ada huruf (besar atau kecil)
                    ->numbers() // Minimal harus ada angka
                    ->uncompromised(), // Cek password yang sudah pernah bocor
                    // Catatan: mixedCase() dan symbols() dihapus untuk fleksibilitas
                    // Password minimal: 8 karakter, mengandung huruf dan angka
            ],
            'terms' => ['required', 'accepted'], // Wajib setuju terms
        ], [
            // Nama
            'name.required' => 'Nama lengkap wajib diisi.',
            'name.string' => 'Nama lengkap harus berupa teks.',
            'name.max' => 'Nama lengkap terlalu panjang. Maksimal 255 karakter.',
            'name.regex' => 'Nama hanya boleh mengandung huruf, angka, spasi, titik, tanda hubung, dan apostrof.',
            
            // Email
            'email.required' => 'Alamat email wajib diisi.',
            'email.string' => 'Alamat email harus berupa teks.',
            'email.lowercase' => 'Alamat email harus menggunakan huruf kecil.',
            'email.email' => 'Format email tidak valid. Pastikan email yang Anda masukkan benar (contoh: nama@email.com).',
            'email.max' => 'Alamat email terlalu panjang. Maksimal 255 karakter.',
            'email.unique' => 'Email ini sudah terdaftar. Silakan gunakan email lain atau masuk dengan akun yang sudah ada.',
            
            // Password
            'password.required' => 'Kata sandi wajib diisi.',
            'password.confirmed' => 'Konfirmasi kata sandi tidak cocok. Pastikan kata sandi dan konfirmasi kata sandi sama.',
            'password.min' => 'Kata sandi minimal 8 karakter. Silakan gunakan kata sandi yang lebih panjang.',
            'password.letters' => 'Kata sandi harus mengandung minimal 1 huruf (a-z atau A-Z).',
            'password.numbers' => 'Kata sandi harus mengandung minimal 1 angka (0-9).',
            'password.uncompromised' => 'Kata sandi ini telah terdeteksi dalam data breach dan tidak aman. Silakan gunakan kata sandi yang lebih kuat dan unik.',
            
            // Terms
            'terms.required' => 'Anda harus menyetujui syarat dan ketentuan untuk melanjutkan pendaftaran.',
            'terms.accepted' => 'Anda harus menyetujui syarat dan ketentuan dengan mencentang kotak persetujuan.',
        ]);

        // Sanitasi input untuk mencegah XSS
        $sanitizedName = htmlspecialchars(strip_tags($validated['name']), ENT_QUOTES, 'UTF-8');
        $sanitizedEmail = filter_var($validated['email'], FILTER_SANITIZE_EMAIL);

        // Pastikan user_type selalu 'user' untuk registrasi publik (prevent privilege escalation)
        $user = User::create([
            'name' => $sanitizedName,
            'email' => $sanitizedEmail,
            'password' => Hash::make($validated['password']),
            'user_type' => 'user', // Hardcode untuk mencegah manipulasi
            'is_active' => true, // Default aktif untuk user baru
            'must_change_password' => false, // Tidak perlu ganti password untuk user baru
        ]);

        event(new Registered($user));

        // Log registrasi untuk audit
        \Log::info('User registered', [
            'user_id' => $user->id,
            'email' => $user->email,
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);

        Auth::login($user);

        return redirect(route('index', absolute: false));
    }
}
