<?php

namespace App\Http\Controllers;

use App\Models\Contact;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;

class AdminContactController extends Controller
{
    public function index(Request $request)
    {
        $query = Contact::query();

        // Search functionality
        if ($request->filled('search') && trim($request->search) !== '') {
            $search = trim($request->search);
            $query->where(function($q) use ($search) {
                $q->where('nama', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('pesan', 'like', "%{$search}%");
            });
        }

        // Filter by status
        if ($request->filled('status') && in_array($request->status, ['unread', 'read'])) {
            if ($request->status === 'unread') {
                $query->where('is_read', false);
            } elseif ($request->status === 'read') {
                $query->where('is_read', true);
            }
        }

        // Urutkan unread (is_read = false) di atas terlebih dahulu, lalu latest created_at
        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $contacts = $query->orderBy('is_read', 'asc') // false (0) di atas
            ->orderBy('created_at', 'desc')
            ->paginate($itemsPerPage)
            ->withQueryString();
            
        return view('admin.contacts.index', compact('contacts'));
    }

    public function destroy(Contact $contact)
    {
        try {
            $contact->delete();
            return redirect()->route('admin.contacts.index')
                ->with('success', 'Pesan kontak berhasil dihapus!');
        } catch (\Exception $e) {
            return redirect()->route('admin.contacts.index')
                ->with('error', 'Gagal menghapus pesan kontak: ' . $e->getMessage());
        }
    }

    public function reply(Request $request, Contact $contact)
    {
        $validator = Validator::make($request->all(), [
            'subject' => 'required|string|max:255',
            'message' => 'required|string',
            'reply_method' => 'required|in:email,whatsapp,phone',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        try {
            // Update contact with admin reply
            $contact->update([
                'admin_reply' => $request->message,
                'admin_replied_at' => now(),
                'reply_method' => $request->reply_method,
                'is_read' => true,
            ]);

            // Send reply based on method
            if ($request->reply_method === 'email') {
                // Kirim email reply
                // Kirim notifikasi ke user
                try {
                    $notificationService = new \App\Services\NotificationService();
                    $notificationService->notifyContactReplied($contact->fresh());
                } catch (\Exception $e) {
                    \Log::error('Failed to send contact replied notification: ' . $e->getMessage());
                }

                Mail::send('emails.contact-reply', [
                    'contact' => $contact,
                    'subject' => $request->subject,
                    'message' => $request->message,
                ], function ($mail) use ($contact, $request) {
                    $mail->to($contact->email, $contact->nama)
                         ->subject('Re: ' . $request->subject)
                         ->from(config('mail.from.address'), config('mail.from.name'));
                });
            } elseif ($request->reply_method === 'whatsapp') {
                // Generate WhatsApp link
                $whatsappMessage = urlencode("Halo {$contact->nama},\n\n{$request->message}\n\nTerima kasih,\nAdmin POLJAM TECH");
                $whatsappLink = "https://wa.me/{$contact->phone}?text={$whatsappMessage}";
                
                return response()->json([
                    'success' => true,
                    'message' => 'Reply berhasil disimpan! Klik link WhatsApp untuk mengirim pesan.',
                    'whatsapp_link' => $whatsappLink,
                ]);
            } elseif ($request->reply_method === 'phone') {
                // Generate phone link
                $phoneLink = "tel:{$contact->phone}";
                
                return response()->json([
                    'success' => true,
                    'message' => 'Reply berhasil disimpan! Klik link untuk menelepon.',
                    'phone_link' => $phoneLink,
                ]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Reply email berhasil dikirim!',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengirim reply: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function markAsRead(Contact $contact)
    {
        try {
            $contact->update(['is_read' => true]);
            
            return response()->json([
                'success' => true,
                'message' => 'Pesan ditandai sebagai dibaca.',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal update status: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function bulkMarkAsRead(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'contact_ids' => 'required|array',
            'contact_ids.*' => 'exists:contacts,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
            ], 422);
        }

        try {
            Contact::whereIn('id', $request->contact_ids)
                ->update(['is_read' => true]);

            return response()->json([
                'success' => true,
                'message' => count($request->contact_ids) . ' pesan berhasil ditandai sebagai dibaca.',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal update status: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function bulkMarkAsUnread(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'contact_ids' => 'required|array',
            'contact_ids.*' => 'exists:contacts,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
            ], 422);
        }

        try {
            Contact::whereIn('id', $request->contact_ids)
                ->update(['is_read' => false]);

            return response()->json([
                'success' => true,
                'message' => count($request->contact_ids) . ' pesan berhasil ditandai sebagai belum dibaca.',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal update status: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function bulkDelete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'contact_ids' => 'required|array',
            'contact_ids.*' => 'exists:contacts,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
            ], 422);
        }

        try {
            Contact::whereIn('id', $request->contact_ids)->delete();

            return response()->json([
                'success' => true,
                'message' => count($request->contact_ids) . ' pesan berhasil dihapus.',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus pesan: ' . $e->getMessage(),
            ], 500);
        }
    }
}
