<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    public function index(Request $request)
    {
        $query = User::query();

        // Filter berdasarkan role (kecuali admin)
        if ($request->filled('role')) {
            $query->where('user_type', $request->role);
        } else {
            // Default: exclude admin dari list
            $query->where('user_type', '!=', 'admin');
        }

        // Filter berdasarkan status
        if ($request->filled('status')) {
            if ($request->status === 'active') {
                $query->active();
            } elseif ($request->status === 'inactive') {
                $query->inactive();
            } elseif ($request->status === 'never_logged_in') {
                $query->neverLoggedIn();
            }
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'name');
        $sortOrder = $request->get('sort_order', 'asc');
        
        $allowedSorts = ['name', 'email', 'user_type', 'is_active', 'last_login_at', 'created_at'];
        if (in_array($sortBy, $allowedSorts)) {
            $query->orderBy($sortBy, $sortOrder);
        } else {
            $query->orderBy('user_type')->orderBy('name');
        }

        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $users = $query->paginate($itemsPerPage)->withQueryString();

        // Statistics
        $stats = [
            'total' => User::where('user_type', '!=', 'admin')->count(),
            'active' => User::where('user_type', '!=', 'admin')->active()->count(),
            'inactive' => User::where('user_type', '!=', 'admin')->inactive()->count(),
            'never_logged_in' => User::where('user_type', '!=', 'admin')->neverLoggedIn()->count(),
            'by_role' => [
                'user' => User::where('user_type', 'user')->count(),
                'teknisi' => User::where('user_type', 'teknisi')->count(),
                'supervisor' => User::where('user_type', 'supervisor')->count(),
            ],
            'new_this_month' => User::where('user_type', '!=', 'admin')
                ->whereMonth('created_at', now()->month)
                ->whereYear('created_at', now()->year)
                ->count(),
        ];

        return view('admin.users.index', compact('users', 'stats'));
    }

    public function create()
    {
        return view('admin.users.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255', 'regex:/^[a-zA-Z0-9\s\.\-\']+$/u'], // Sanitasi nama
            'email' => ['required', 'string', 'email:rfc,dns', 'max:255', 'unique:users'], // Validasi email dengan DNS check
            'password' => [
                'required',
                'string',
                'min:8',
                'confirmed',
                'regex:/^(?=.*[a-zA-Z])(?=.*\d)/', // Password minimal: huruf (besar atau kecil) dan angka
                // Catatan: Huruf besar dan simbol tidak wajib untuk fleksibilitas
            ],
            'user_type' => ['required', 'string', Rule::in(['user', 'teknisi', 'supervisor'])],
            'phone' => ['nullable', 'string', 'max:20', 'regex:/^[0-9\-\+\s\(\)]+$/'], // Validasi format nomor telepon
            'address' => ['nullable', 'string', 'max:500'],
            'postal_code' => ['nullable', 'string', 'max:10', 'regex:/^[0-9]+$/'], // Hanya angka untuk kode pos
            'photo' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif,webp', 'max:2048'],
        ], [
            'name.regex' => 'Nama hanya boleh mengandung huruf, angka, spasi, titik, tanda hubung, dan apostrof.',
            'email.email' => 'Format email tidak valid.',
            'password.regex' => 'Password harus mengandung minimal 1 huruf (besar atau kecil) dan 1 angka.',
            'phone.regex' => 'Format nomor telepon tidak valid.',
            'postal_code.regex' => 'Kode pos hanya boleh mengandung angka.',
        ]);

        // Defense in depth: Prevent creating admin user (meskipun sudah ada di validasi Rule::in)
        if ($validated['user_type'] === 'admin') {
            abort(403, 'Tidak dapat membuat user dengan role Admin');
        }

        // Sanitasi input untuk mencegah XSS
        $validated['name'] = htmlspecialchars(strip_tags($validated['name']), ENT_QUOTES, 'UTF-8');
        $validated['email'] = filter_var($validated['email'], FILTER_SANITIZE_EMAIL);
        if (isset($validated['phone'])) {
            $validated['phone'] = filter_var($validated['phone'], FILTER_SANITIZE_STRING);
        }
        if (isset($validated['address'])) {
            $validated['address'] = htmlspecialchars(strip_tags($validated['address']), ENT_QUOTES, 'UTF-8');
        }

        // Simpan password plain text untuk dikirim via email (sebelum di-hash)
        $plainPassword = $validated['password'];

        // Hash password
        $validated['password'] = Hash::make($validated['password']);

        // Handle photo upload dengan validasi tambahan
        if ($request->hasFile('photo')) {
            $file = $request->file('photo');
            // Validasi tambahan untuk mencegah upload file berbahaya
            if ($file->isValid() && in_array($file->getClientOriginalExtension(), ['jpeg', 'png', 'jpg', 'gif', 'webp'])) {
                $validated['photo'] = $file->store('profiles', 'uploads');
            } else {
                return back()->withErrors(['photo' => 'Format file tidak valid.'])->withInput();
            }
        }

        // Set default values secara eksplisit (prevent mass assignment vulnerability)
        $userData = [
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => $validated['password'],
            'user_type' => $validated['user_type'],
            'is_active' => true,
            'must_change_password' => true, // User wajib ganti password saat login pertama kali untuk keamanan
        ];

        // Tambahkan field opsional jika ada
        if (isset($validated['phone'])) {
            $userData['phone'] = $validated['phone'];
        }
        if (isset($validated['address'])) {
            $userData['address'] = $validated['address'];
        }
        if (isset($validated['postal_code'])) {
            $userData['postal_code'] = $validated['postal_code'];
        }
        if (isset($validated['photo'])) {
            $userData['photo'] = $validated['photo'];
        }

        // Create user dengan data yang sudah disanitasi
        $user = User::create($userData);

        // Log aktivitas untuk audit
        \Log::info('User created by admin', [
            'created_by' => auth()->id(),
            'user_id' => $user->id,
            'email' => $user->email,
            'user_type' => $user->user_type,
            'ip_address' => $request->ip(),
        ]);

        // Kirim email notifikasi dengan kredensial login
        try {
            \Illuminate\Support\Facades\Mail::to($user->email)->send(
                new \App\Mail\NewUserCreatedByAdmin($user, $plainPassword)
            );
        } catch (\Exception $e) {
            \Log::error('Failed to send new user creation email', [
                'user_id' => $user->id,
                'email' => $user->email,
                'error' => $e->getMessage()
            ]);
            // Tetap lanjutkan meskipun email gagal dikirim
        }

        return redirect()->route('admin.users.index')
            ->with('add_message', "User <strong>{$user->name}</strong> berhasil ditambahkan! Kredensial login telah dikirim ke email: <strong>{$user->email}</strong>. User wajib mengganti password saat login pertama kali.");
    }

    public function show($id)
    {
        $user = User::findOrFail($id);

        // Prevent viewing admin (optional - bisa dihapus jika ingin admin bisa lihat detail admin lain)
        // if ($user->isAdmin()) {
        //     abort(403, 'Tidak dapat melihat detail user dengan role Admin');
        // }

        return view('admin.users.show', compact('user'));
    }

    public function edit($id)
    {
        $user = User::findOrFail($id);

        // Prevent editing admin
        if ($user->isAdmin()) {
            abort(403, 'Tidak dapat mengedit user dengan role Admin');
        }

        return view('admin.users.edit', compact('user'));
    }

    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);

        // Prevent editing admin
        if ($user->isAdmin()) {
            abort(403, 'Tidak dapat mengedit user dengan role Admin');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
            'user_type' => ['required', 'string', Rule::in(['user', 'teknisi', 'supervisor'])],
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:500',
            'postal_code' => 'nullable|string|max:10',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        ]);

        // Password tidak bisa diubah melalui form edit untuk keamanan
        // Gunakan fitur Reset Password untuk mereset password user

        // Handle photo upload
        if ($request->hasFile('photo')) {
            // Delete old photo if exists
            if ($user->photo && Storage::disk('uploads')->exists('profiles/' . $user->photo)) {
                Storage::disk('uploads')->delete('profiles/' . $user->photo);
            }
            $validated['photo'] = $request->file('photo')->store('profiles', 'uploads');
        }

        $user->update($validated);

        return redirect()->route('admin.users.index')
            ->with('add_message', 'User berhasil diperbarui!');
    }

    public function destroy($id)
    {
        $user = User::findOrFail($id);

        // Prevent deleting admin
        if ($user->isAdmin()) {
            abort(403, 'Tidak dapat menghapus user dengan role Admin');
        }

        // Delete photo if exists
        if ($user->photo && Storage::disk('uploads')->exists('profiles/' . $user->photo)) {
            Storage::disk('uploads')->delete('profiles/' . $user->photo);
        }

        $user->delete();

        return redirect()->route('admin.users.index')
            ->with('add_message', 'User berhasil dihapus!');
    }

    public function resetPassword($id)
    {
        $user = User::findOrFail($id);

        // Prevent resetting admin password
        if ($user->isAdmin()) {
            abort(403, 'Tidak dapat mereset password user dengan role Admin');
        }

        // Generate random password (12 karakter dengan kombinasi huruf dan angka)
        $newPassword = \Str::random(12);
        
        // Update password dan set flag must_change_password
        $user->password = Hash::make($newPassword);
        $user->must_change_password = true; // User wajib ganti password saat login
        $user->save();

        // Kirim email notifikasi ke user
        try {
            \Illuminate\Support\Facades\Mail::to($user->email)->send(
                new \App\Mail\PasswordResetByAdmin($user, $newPassword)
            );
        } catch (\Exception $e) {
            \Log::error('Failed to send password reset email', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);
        }

        return redirect()->route('admin.users.index')
            ->with('add_message', "Password user <strong>{$user->name}</strong> berhasil direset! Password baru telah dikirim ke email: <strong>{$user->email}</strong>. User wajib mengganti password saat login pertama kali.");
    }

    public function toggleStatus($id)
    {
        $user = User::findOrFail($id);

        // Prevent toggling admin status
        if ($user->isAdmin()) {
            abort(403, 'Tidak dapat mengubah status user dengan role Admin');
        }

        $user->is_active = !$user->is_active;
        $user->save();

        $status = $user->is_active ? 'diaktifkan' : 'dinonaktifkan';
        return redirect()->route('admin.users.index')
            ->with('add_message', "User {$user->name} berhasil {$status}!");
    }

    public function bulkAction(Request $request)
    {
        $request->validate([
            'action' => 'required|in:activate,deactivate,delete',
            'user_ids' => 'required|array',
            'user_ids.*' => 'exists:users,id',
        ]);

        $userIds = $request->user_ids;
        $users = User::whereIn('id', $userIds)
            ->where('user_type', '!=', 'admin')
            ->get();

        $count = 0;

        switch ($request->action) {
            case 'activate':
                foreach ($users as $user) {
                    $user->is_active = true;
                    $user->save();
                    $count++;
                }
                $message = "{$count} user berhasil diaktifkan!";
                break;

            case 'deactivate':
                foreach ($users as $user) {
                    $user->is_active = false;
                    $user->save();
                    $count++;
                }
                $message = "{$count} user berhasil dinonaktifkan!";
                break;

            case 'delete':
                foreach ($users as $user) {
                    // Delete photo if exists
                    if ($user->photo && Storage::disk('uploads')->exists('profiles/' . $user->photo)) {
                        Storage::disk('uploads')->delete('profiles/' . $user->photo);
                    }
                    $user->delete();
                    $count++;
                }
                $message = "{$count} user berhasil dihapus!";
                break;
        }

        return redirect()->route('admin.users.index')
            ->with('add_message', $message);
    }
}

