<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\StockOpname;
use App\Models\StockOpnameItem;
use App\Models\Material;
use App\Models\Sparepart;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class StockOpnameController extends Controller
{
    /**
     * Tampilkan daftar stock opname
     */
    public function index(Request $request)
    {
        // Mulai query
        $query = StockOpname::with('creator', 'approver')->orderBy('opname_date', 'desc');

        // Filter: search by opname_number - perbaiki dengan trim dan ID search
        if ($request->filled('search') && trim($request->search) !== '') {
            $searchTerm = trim($request->search);
            $query->where(function ($q) use ($searchTerm) {
                $q->where('opname_number', 'like', "%{$searchTerm}%");
                // Tambahkan pencarian ID jika search adalah numerik
                if (is_numeric($searchTerm)) {
                    $q->orWhere('id', $searchTerm);
                }
            });
        }

        // Filter: status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter: tanggal mulai
        if ($request->filled('start_date')) {
            $query->whereDate('opname_date', '>=', $request->start_date);
        }

        // Filter: tanggal akhir
        if ($request->filled('end_date')) {
            $query->whereDate('opname_date', '<=', $request->end_date);
        }

        // Pagination
        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $stockOpnames = $query->paginate($itemsPerPage)->withQueryString();

        return view('admin.backoffice.stock-opname.index', compact('stockOpnames'));
    }


    /**
     * Tampilkan form buat stock opname baru
     */
    public function create()
    {
        $materials = Material::all()->map(function ($m) {
            return [
                'id' => $m->id,
                'name' => $m->name,
                'code' => $m->material_code ?? '-',
                'stock' => $m->stock,
                'unit' => $m->unit
            ];
        });

        $spareparts = Sparepart::all()->map(function ($s) {
            return [
                'id' => $s->id,
                'name' => $s->name,
                'code' => $s->part_code ?? '-',
                'stock' => $s->stock,
                'unit' => $s->unit
            ];
        });

        return view('admin.backoffice.stock-opname.create', compact('materials', 'spareparts'));
    }


    /**
     * Simpan stock opname baru
     */
    public function store(Request $request)
    {
        $request->validate([
            'opname_date' => 'required|date',
            'notes' => 'nullable|string',
            'items' => 'required|array',
            'items.*.itemable_type' => 'required|string',
            'items.*.itemable_id' => 'required|integer',
            'items.*.physical_qty' => 'required|numeric|min:0',
            'items.*.reason' => 'nullable|string',
        ]);

        DB::transaction(function () use ($request) {
            // Generate nomor stock opname otomatis: SO-YYYYMM-XXX
            $lastOpname = StockOpname::orderBy('id', 'desc')->first();
            $number = 'SO-' . date('Ym') . '-' . str_pad(optional($lastOpname)->id + 1 ?? 1, 3, '0', STR_PAD_LEFT);

            $stockOpname = StockOpname::create([
                'opname_number' => $number,
                'opname_date' => $request->opname_date,
                'notes' => $request->notes,
                'status' => 'draft',
                'created_by' => Auth::id(),
            ]);

            // Simpan detail items
            foreach ($request->items as $item) {
                // Validasi itemable_type
                $modelClass = $item['itemable_type'];

                // Pastikan model class valid
                if (!in_array($modelClass, ['App\\Models\\Material', 'App\\Models\\Sparepart'])) {
                    continue;
                }

                // Cari item berdasarkan ID dan type
                $itemable = $modelClass::findOrFail($item['itemable_id']);

                // Simpan item stock opname
                $stockOpname->items()->create([
                    'itemable_id' => $itemable->id,
                    'itemable_type' => $modelClass,
                    'item_name' => $itemable->name,
                    'item_code' => $itemable->material_code ?? $itemable->part_code ?? '-',
                    'unit' => $itemable->unit,
                    'system_qty' => $itemable->stock, // Stok dari database saat ini
                    'physical_qty' => $item['physical_qty'], // Stok fisik yang diinput user
                    // difference akan dihitung otomatis via model accessor/mutator
                    'reason' => $item['reason'] ?? null,
                    'notes' => $item['notes'] ?? null,
                ]);
            }
        });

        return redirect()->route('admin.backoffice.stock-opname.index')
            ->with('success', 'Stock opname berhasil dibuat.');
    }

    /**
     * Tampilkan detail stock opname
     */
    public function show($id)
    {
        $stockOpname = StockOpname::with('items.itemable', 'creator', 'approver')
            ->findOrFail($id);

        return view('admin.backoffice.stock-opname.show', compact('stockOpname'));
    }


    /**
     * Tampilkan form edit stock opname (hanya draft)
     */
    public function edit($id)
    {
        $stockOpname = StockOpname::with('items.itemable')->findOrFail($id);

        if ($stockOpname->status !== 'draft') {
            return redirect()->back()->with('error', 'Hanya stock opname draft yang bisa diedit.');
        }

        $materials = Material::all()->map(function ($m) {
            return [
                'id' => $m->id,
                'name' => $m->name,
                'code' => $m->material_code ?? '-',
                'stock' => $m->stock,
                'unit' => $m->unit
            ];
        });

        $spareparts = Sparepart::all()->map(function ($s) {
            return [
                'id' => $s->id,
                'name' => $s->name,
                'code' => $s->part_code ?? '-',
                'stock' => $s->stock,
                'unit' => $s->unit
            ];
        });

        return view('admin.backoffice.stock-opname.edit', compact('stockOpname', 'materials', 'spareparts'));
    }

    /**
     * Update stock opname (hanya draft)
     */
    public function update(Request $request, $id)
    {
        $stockOpname = StockOpname::with('items')->findOrFail($id);

        if ($stockOpname->status !== 'draft') {
            return redirect()->back()->with('error', 'Hanya stock opname draft yang bisa diupdate.');
        }

        $request->validate([
            'opname_date' => 'required|date',
            'notes' => 'nullable|string',
            'items' => 'required|array',
            'items.*.itemable_type' => 'required|string',
            'items.*.itemable_id' => 'required|integer',
            'items.*.physical_qty' => 'required|numeric|min:0',
        ]);

        DB::transaction(function () use ($request, $stockOpname) {
            // Update stock opname header
            $stockOpname->update([
                'opname_date' => $request->opname_date,
                'notes' => $request->notes,
            ]);

            $existingItemIds = $stockOpname->items->pluck('id')->toArray();
            $submittedItemIds = [];

            foreach ($request->items as $item) {
                $modelClass = $item['itemable_type'];

                if (!in_array($modelClass, ['App\\Models\\Material', 'App\\Models\\Sparepart'])) {
                    continue;
                }

                $itemable = $modelClass::findOrFail($item['itemable_id']);

                // Jika ada ID item lama (update)
                if (isset($item['id']) && in_array($item['id'], $existingItemIds)) {
                    $stockItem = StockOpnameItem::findOrFail($item['id']);
                    $stockItem->update([
                        'itemable_id' => $itemable->id,
                        'itemable_type' => $modelClass,
                        'item_name' => $itemable->name,
                        'item_code' => $itemable->material_code ?? $itemable->part_code ?? '-',
                        'unit' => $itemable->unit,
                        'system_qty' => $itemable->stock,
                        'physical_qty' => $item['physical_qty'],
                        'reason' => $item['reason'] ?? null,
                        'notes' => $item['notes'] ?? null,
                    ]);
                    $submittedItemIds[] = $stockItem->id;
                } else {
                    // Baru
                    $newItem = $stockOpname->items()->create([
                        'itemable_id' => $itemable->id,
                        'itemable_type' => $modelClass,
                        'item_name' => $itemable->name,
                        'item_code' => $itemable->material_code ?? $itemable->part_code ?? '-',
                        'unit' => $itemable->unit,
                        'system_qty' => $itemable->stock,
                        'physical_qty' => $item['physical_qty'],
                        'reason' => $item['reason'] ?? null,
                        'notes' => $item['notes'] ?? null,
                    ]);
                    $submittedItemIds[] = $newItem->id;
                }
            }

            // Hapus item yang sudah tidak dikirim
            $itemsToDelete = array_diff($existingItemIds, $submittedItemIds);
            if (!empty($itemsToDelete)) {
                StockOpnameItem::whereIn('id', $itemsToDelete)->delete();
            }
        });

        return redirect()->route('admin.backoffice.stock-opname.index')
            ->with('success', 'Stock opname berhasil diupdate.');
    }

    /**
     * Hapus stock opname (soft delete)
     */
    public function destroy($id)
    {
        $stockOpname = StockOpname::findOrFail($id);

        if ($stockOpname->status === 'completed') {
            return redirect()->back()->with('error', 'Stock opname yang sudah selesai tidak bisa dihapus.');
        }

        $stockOpname->delete();

        return redirect()->route('admin.backoffice.stock-opname.index')
            ->with('success', 'Stock opname berhasil dihapus.');
    }

    /**
     * Approve stock opname
     */
    public function approve($id)
    {
        $stockOpname = StockOpname::with('items')->findOrFail($id);

        if ($stockOpname->status !== 'draft') {
            return redirect()->back()->with('error', 'Stock opname hanya bisa diapprove dari draft.');
        }

        DB::transaction(function () use ($stockOpname) {
            $stockOpname->update([
                'status' => 'completed',
                'approved_by' => Auth::id(),
                'approved_at' => now(),
            ]);

            // Update stok master sesuai physical_qty
            foreach ($stockOpname->items as $item) {
                $model = $item->itemable_type::find($item->itemable_id);
                if ($model) {
                    $model->update(['stock' => $item->physical_qty]);
                }
            }
        });

        return redirect()->route('admin.backoffice.stock-opname.index')
            ->with('success', 'Stock opname berhasil diapprove.');
    }

    /**
     * API untuk mengambil semua material + sparepart
     * Digunakan di form create/edit untuk dropdown
     */
    public function getItems()
    {
        $materials = Material::where('is_active', true)->get(['id', 'name', 'stock', 'unit', 'material_code']);
        $spareparts = Sparepart::where('is_active', true)->get(['id', 'name', 'stock', 'unit', 'part_code']);

        return response()->json([
            'materials' => $materials,
            'spareparts' => $spareparts,
        ]);
    }
}
