<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Project;
use App\Models\Setting;
use App\Models\Order;
use Illuminate\Http\Request;

class ProjectController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if (Project::count() === 0) {
            $acceptedOrders = Order::where('order_status', 'diterima')
                ->whereNotNull('product_id')
                ->get();

            foreach ($acceptedOrders as $ord) {
                $ord->ensureProjectExists();
            }
        }

        $query = Project::query();

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('kode_bantu', 'like', "%{$search}%")
                    ->orWhere('nama_proyek', 'like', "%{$search}%");
            });
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by is_active
        if ($request->filled('is_active')) {
            $query->where('is_active', $request->is_active);
        }

        // Get items per page from settings
        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $projects = $query->orderBy('sort_order')->orderBy('kode_bantu')->paginate($itemsPerPage)->withQueryString();

        return view('admin.projects.index', compact('projects'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.projects.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'kode_bantu' => 'required|string|max:50|unique:projects,kode_bantu',
            'nama_proyek' => 'required|string|max:255',
            'tabel_bantuan' => 'nullable|string|max:50',
            'nilai_kontrak' => 'nullable|numeric|min:0',
            'tanggal_mulai' => 'nullable|date',
            'tanggal_selesai' => 'nullable|date|after_or_equal:tanggal_mulai',
            'status' => 'required|in:aktif,selesai,dibatalkan',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer|min:0',
            'keterangan' => 'nullable|string',
        ]);

        // Set tabel_bantuan default ke kode_bantu jika kosong
        if (empty($validated['tabel_bantuan'])) {
            $validated['tabel_bantuan'] = $validated['kode_bantu'];
        }

        Project::create($validated);

        return redirect()->route('admin.projects.index')
            ->with('success', 'Project berhasil ditambahkan');
    }

    /**
     * Display the specified resource.
     */
    public function show(Project $project)
    {
        $project->loadCount('journalEntries');
        return view('admin.projects.show', compact('project'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Project $project)
    {
        return view('admin.projects.edit', compact('project'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Project $project)
    {
        $validated = $request->validate([
            'kode_bantu' => 'required|string|max:50|unique:projects,kode_bantu,' . $project->id,
            'nama_proyek' => 'required|string|max:255',
            'tabel_bantuan' => 'nullable|string|max:50',
            'nilai_kontrak' => 'nullable|numeric|min:0',
            'tanggal_mulai' => 'nullable|date',
            'tanggal_selesai' => 'nullable|date|after_or_equal:tanggal_mulai',
            'status' => 'required|in:aktif,selesai,dibatalkan',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer|min:0',
            'keterangan' => 'nullable|string',
        ]);

        // Set tabel_bantuan default ke kode_bantu jika kosong
        if (empty($validated['tabel_bantuan'])) {
            $validated['tabel_bantuan'] = $validated['kode_bantu'];
        }

        $project->update($validated);

        return redirect()->route('admin.projects.index')
            ->with('success', 'Project berhasil diperbarui');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Project $project)
    {
        // Cek apakah ada journal entries yang menggunakan proyek ini
        if ($project->journalEntries()->count() > 0) {
            return redirect()->back()
                ->with('error', 'Tidak dapat menghapus proyek yang sudah digunakan dalam jurnal');
        }

        $project->delete();

        return redirect()->route('admin.projects.index')
            ->with('success', 'Project berhasil dihapus');
    }
}
